<?php

namespace App\Http\Controllers;

use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\Billings\Order;
use App\Models\PaymentGateway;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;
use App\Jobs\Payments\PaymentProcessJob;
use App\Services\Payments\PartialPaymentService;

class TripayController extends Controller
{
    public function __invoke(Request $request, PartialPaymentService $partialPaymentService)
    {
        Log::info('Tripay call back');
        $signature = $request->header('X-Callback-Signature');
        if (!$signature) return [
            'success' => false,
            'message' => 'Unauthorized'
        ];

        $callBackEvent = $request->header('X-Callback-Event');
        if ($callBackEvent != 'payment_status') return [
            'success' => false,
            'message' => 'Not payment status.'
        ];

        $signingSecret = env('TRIPAY_MODE') === 'development' ? env('TRIPAY_DEV_PRIVATE_KEY') : env('TRIPAY_PRODUCTION_PRIVATE_KEY');

        $computedSignature = hash_hmac('sha256', $request->getContent(), $signingSecret);
        if (!hash_equals($signature, $computedSignature)) return [
            'success' => false,
            'message' => 'Unauthorized'
        ];

        if ($request->is_closed_payment != 1) return [
            'success' => false,
            'message' => 'Not closed payment'
        ];

        try {
            Log::info($request->all());
            $order = Order::whereMerchantRef($request->merchant_ref)
                ->wherePaymentGatewayChannel('tripay')
                ->wherePaymentMethod($request->payment_method_code)
                ->whereAmount($request->total_amount)
                ->first();

            if (!$order) return [
                'success' => false,
                'message' => 'Order not found'
            ];
            $status = Str::lower($request->status);

            switch ($status) {
                // handle status PAID
                case 'paid':
                    // $status = 'paid';
                    $message = 'Payment successfully';
                    break;
                // handle status EXPIRED
                case 'expired':
                    // $status = 'expired';
                    $message = 'Payment expired';
                    break;
                // handle status FAILED
                case 'failed':
                    // $status = 'failed';
                    $message = 'Payment failed';
                    break;
                case 'refund':
                    // $status = 'refund';
                    $message = 'Payment refund';
                    break;
                case 'unpaid':
                    // $status = 'unpaid';
                    $message = 'Payment unpaid';
                    break;
                default:
                    return [
                        'success' => false,
                        'message' => 'Unrecognized payment status'
                    ];
            }

            $invoice = $order->invoice;
            if ($status != 'paid') {
                $order->forceFill(['status' => $status])->save();
                $invoiceExpired = Carbon::parse($invoice->due_date)->isPast();
                $invoice->forceFill(['status' => $invoiceExpired ? 'overdue' : 'unpaid'])->save();
                return [
                    'success' => true,
                    'message' => 'Payment status ' . $status
                ];
            }

            $ammountReceived = $request->amount_received ?? $request->total_amount - $request->total_fee + $request->fee_merchant;

            /*
            $result = $partialPaymentService->processPartialPayment(
                $invoice,
                $ammountReceived,
                'tripay',
                null,
                'Tripay'
            );

            if ($result['success']){
                $order->forceFill([
                    'status' => $status
                ])->save();
                return [
                    'success' => true,
                    'message' => 'Payment successfully.'
                ];
            }

            return [
                'success' => false,
                'message' => 'Bad request'
            ];*/
            dispatch(new PaymentProcessJob(
                $invoice,
                $ammountReceived,
                'tripay',
                null,
                'Tripay'
            ))->onQueue('default');

        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
}
