<?php

namespace App\Livewire\Admin\Billings;

use DateTime;
use App\Models\User;
use Livewire\Component;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use Illuminate\Support\Carbon;
use Livewire\Attributes\Title;
use App\Models\Billings\Invoice;
use Illuminate\Support\Facades\DB;
use App\Services\Billings\ExportInvoiceService;
use App\Services\WhatsappGateway\WhatsappNotificationService;
use App\Traits\NotificationTrait;

class BillingManagement extends Component
{
    use WithPagination, NotificationTrait;
    //Short by
    public $sortField = 'due_date';
    public $sortDirection = 'asc';
    protected $queryString = ['sortField', 'sortDirection'];

    // Pagination
    public $perPage = 25;

    //Search
    public $search_name = '';
    public $search_address = '';
    public $search_with_status = 'pending';
    public $search_with_month = "all-month";
    public $search_with_year = "all-year";
    public $search_with_teller;
    public $startDateDeadline = '';
    public $endDateDeadline;
    public $periodeMonths;
    public $selectedInvoice = [];
    public $checkAll;

    #[Title('Billing')]
    /**
     * Sort by function
     */
    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection == 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    public function updatedSearchName()
    {
        $this->resetPage();
    }

    public function updatedSearchAddress()
    {
        $this->resetPage();
    }

    public function updatedSearchWithTeller()
    {
        $this->resetPage();
    }

    public function updatedEndDateDeadline()
    {
        $this->resetPage();
    }

    public function updatedStartDateDeadline()
    {
        $this->resetPage();
    }

    public function bulkSelectedPayment()
    {
        $this->dispatch('bulk-payment-modal', invoiceSelected: $this->selectedInvoice);
    }

    public function bulkDeleteSelected()
    {
        $this->dispatch('bulk-delete-invoice-modal', invoiceSelected: $this->selectedInvoice);
    }
    public function clearSearch()
    {
        $this->startDateDeadline = null;
        $this->endDateDeadline = null;
        $this->search_with_year = "all-year";
        $this->search_with_month = "all-month";
        $this->search_with_status = 'pending';
        // $this->search_with_status = null;
        $this->search_name = null;
        $this->search_address = null;
        $this->search_with_teller = null;
    }

    public function updatedSearchWithStatus($status)
    {
        // $this->resetPage();
        if ($status == "paid") {
            $getMonthPeriode =  $this->getMonthsPeriode(Carbon::now()->format('Y'));
            $this->periodeMonths = $getMonthPeriode->get();
            if ($this->periodeMonths->count()) {
                $lastMonth =  DateTime::createFromFormat('m', $getMonthPeriode->latest('periode')->first()->month);
                $this->search_with_month = Carbon::parse($lastMonth)->format('m');
            }

            $this->search_with_year = Carbon::now()->format('Y');
        } else {
            $this->search_with_month = 'all-month';
            $this->search_with_year = 'all-year';
        }
    }

    public function updatedSearchWithYear($year)
    {
        //$this->resetPage();
        if ($year == 'all-year') {
            $this->search_with_month = 'all-month';
        } else {
            if ($this->search_with_status == "paid") {
                $getMonthPeriode =  $this->getMonthsPeriode($year);
                if ($getMonthPeriode->count()) {
                    $lastMonth =  DateTime::createFromFormat('!m', $getMonthPeriode->latest('periode')->first()->month);
                    $this->search_with_month = Carbon::parse($lastMonth)->format('m');
                }
            }
        }
    }

    public function updatedCheckAll($checkAll)
    {
        // $this->resetPage();
        if (!$checkAll) {
            $this->selectedInvoice = [];
        }
    }

    private function getMonthsPeriode($year)
    {
        return Invoice::whereYear('periode', $year)->selectRaw('extract(month FROM periode) AS month')
            ->distinct()
            ->orderBy('month', 'desc');
    }

    public function download_customer_invoice($invoice, ExportInvoiceService $exportInvoiceService)
    {
        $invoice = Invoice::findOrFail($invoice);
        $response = $exportInvoiceService->download_invoice_file($invoice, 'invoice_files');
        if ($response) {
            return $response;
        } else {
            $this->error_notification('Failed', 'Invoice file not found.');
        }
    }

    public function download_customer_invoices(User $user, ExportInvoiceService $exportInvoiceService)
    {
        // $address = $user->user_address->address;
        // $address = $address === null ? '_' : '_' . $address;
        //  $fileName =  Str::slug('invoice_' . $user->full_name . $address, '_');

        // $invoicesFile = $exportInvoiceService->create_invoices_file(
        //     collect([$user]),
        //     $fileName
        // );


        // $response = $exportInvoiceService->download($invoicesFile);
        // if ($response) {
        //     return $response;
        // }

        // $invoices = Invoice::where('status', '!=', 'paid')->get();
        // $startInvoice = $invoices->latest()->periode;
        //// dd(Carbon::parse($startInvoice)->format('m-Y'));
        //$invoicesFile = $exportInvoiceService->create_invoices_file($invoices);
    }

    public function sendNotification(User $user, WhatsappNotificationService $whatsappNotificationService)
    {
        $customerPakets = $user->customer_pakets;
        foreach ($customerPakets as $customerPaket) {
            $lastInvoice = $customerPaket->invoices()->latest()->first();
            $nowDate = new DateTime(Carbon::now()->startOfDay());
            $deadline = new DateTime($lastInvoice->due_date);
            $interval_day = $nowDate->diff($deadline)->format('%a');
            if (Carbon::now()->gt($lastInvoice->due_date) && $interval_day != 0) {
                $interval_day = '+' . $interval_day;
            }
            $whatsappNotificationService->sendUpcomingDueReminders($customerPaket, $interval_day);
        }
    }

    public function cancelPaylater(Invoice $invoice)
    {
        $customerPaket = $invoice->customer_paket;
        $customerPaket->forceFill([
            //'paylater_date' => $customerPaket->customer_paket_status_latest->paylater_date
            'paylater_date' => null
        ])->save();
    }

    #[On('refresh-selected-billing-paket')]
    public function refreshSelectedCustomerPaket()
    {
        $this->selectedInvoice = [];
        $this->checkAll = false;
    }

    #[On('refresh-billing-paket')]
    public function render()
    {
        $invoices = Invoice::join('customer_pakets', 'customer_pakets.id', 'invoices.customer_paket_id')
            ->join('users', 'users.id', 'customer_pakets.user_id')
            ->join('customer_paket_addresses', 'customer_pakets.id', 'customer_paket_addresses.customer_paket_id')
            ->where('address_type', 'billing-address')
            ->select(
                DB::raw("CONCAT(users.first_name,' ',COALESCE(users.last_name,'')) as full_name"),
                DB::raw("CONCAT(customer_paket_addresses.address, ' ', customer_paket_addresses.subdistrict, ' ', customer_paket_addresses.district, ' ', customer_paket_addresses.city, ' ', customer_paket_addresses.province) as full_address"),
                'invoices.*',
                'users.username as user_username',
            )

            ->when($this->search_with_status, function ($builder) {
                if ($this->search_with_status == "pending") {
                    $builder->where('invoices.status', '!=', 'paid');
                } elseif ($this->search_with_status == "paylater") {
                    $builder->whereNotNull('customer_pakets.paylater_date');
                } else {
                    $builder->where('invoices.status', $this->search_with_status);
                }
            })
            ->when($this->search_with_month, function ($builder) {
                $builder->where(function ($builder) {
                    if ($this->search_with_month != "all-month") {
                        $builder->whereMonth('periode', $this->search_with_month);
                    }
                });
            }, function ($builder) {
                if ($this->search_with_status == "paid") {
                    $this->search_with_month = Carbon::now()->format('m');
                    return $builder->whereMonth('periode', Carbon::now()->format('m'));
                }
            })
            ->when($this->search_with_year, function ($builder) {
                $builder->where(function ($builder) {
                    if ($this->search_with_year != "all-year") {
                        $builder->whereYear('periode', $this->search_with_year);
                    }
                });
            }, function ($builder) {
                if ($this->search_with_status == "paid") {
                    $this->search_with_year = Carbon::now()->format('Y');
                    return $builder->whereYear('periode', $this->search_with_year);
                }
            })
            ->when($this->startDateDeadline, function ($builder) {
                $this->search_with_month = "all-month";
                $this->search_with_year = "all-year";
                if ($this->startDateDeadline && $this->endDateDeadline) {
                    if ($this->search_with_status == "paid") {
                        $builder->whereBetween('paid_at', [Carbon::parse($this->startDateDeadline)->startOfDay(), Carbon::parse($this->endDateDeadline)->endOfDay()]);
                    } else {
                        $builder->whereBetween('due_date', [Carbon::parse($this->startDateDeadline)->startOfDay(), Carbon::parse($this->endDateDeadline)->endOfDay()]);
                    }
                } else {
                    if ($this->search_with_status == "paid") {
                        $builder->whereBetween('paid_at', [Carbon::parse($this->startDateDeadline)->startOfDay(), Carbon::now()->endOfDay()]);
                    } else {
                        $builder->whereBetween('due_date', [Carbon::parse($this->startDateDeadline)->startOfDay(), Carbon::now()->endOfDay()]);
                    }
                }
            })
            ->when($this->search_with_teller, function ($builder) {
                $builder->with(['payments' => function ($builder) {
                    $builder->where('teller', $this->search_with_teller);
                }]);
            })
            ->when($this->search_name, function ($builder) {
                $sql = "CONCAT(users.first_name,' ',COALESCE(users.last_name,''))  like ?";
                $builder->whereRaw($sql,  "%" . $this->search_name . "%");
            })
            ->when($this->search_address, function ($builder) {
                $sql = "CONCAT(customer_paket_addresses.address, ' ', customer_paket_addresses.subdistrict, ' ', customer_paket_addresses.district, ' ', customer_paket_addresses.city, ' ', customer_paket_addresses.province)  like ?";
                $builder->whereRaw($sql,  "%" . $this->search_address . "%");
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        $users = User::role('admin')->get();
        return view('livewire.admin.billings.billing-management', [
            'invoices' => $invoices,
            'users' => $users,
        ]);
    }
}
