<?php

namespace App\Livewire\Admin\Billings\Modal;

use App\Traits\NotificationTrait;
use Livewire\Component;
use Livewire\Attributes\On;
use Illuminate\Support\Carbon;
use App\Models\Billings\Invoice;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Services\Billings\BillingService;
use App\Http\Requests\CurrentPasswordRequest;
use App\Livewire\Actions\Billings\InvoiceAction;

class DeleteInvoice extends Component
{
    use NotificationTrait;

    public $showDeleteInvoiceModal = false;
    public $invoice;
    public $input = [];


    #[On('show-delete-invoice-modal')]
    public function showDeleteInvoice($invoice)
    {
        if (Auth::user()->can('invoice action') || Auth::user()->hasRole('admin')) {
            $invoice = Invoice::findOrFail($invoice);
            $this->showDeleteInvoiceModal = true;
            $this->invoice = $invoice;
        } else {
            $this->error_notification('Unauthorized', 'You dont have permission.');
        }
    }

    public function deleteInvoice(CurrentPasswordRequest $request, BillingService $billingService)
    {
        $this->resetErrorBag();
        $request->validate($this->input);
        DB::beginTransaction();
        try {
            //Last Invoice
            $customerPaket = $this->invoice->customer_paket;
            $latestInvoice = $customerPaket->invoices()->latest('periode')->first();
            $renewalPeriod = $customerPaket->getRenewalPeriod();

            if ($latestInvoice) {
                $expiredDate = $latestInvoice->start_periode;
                $startDate = Carbon::parse($expiredDate)->sub($renewalPeriod);
            } else {
                $startDate = $customerPaket->start_date;
                $expiredDate = Carbon::parse($startDate)->add($renewalPeriod);
            }
            $previouslyBilledAt = Carbon::parse($customerPaket->next_billed_at)->sub($renewalPeriod);
            $customerPaket->forceFill([
                'start_date' => $startDate,
                'expired_date' => $expiredDate,
                'next_billed_at' => $previouslyBilledAt,
            ])->save();

            (new InvoiceAction)->delete_billing($this->invoice);
            $title = trans('Success');
            $message = trans('Delete invoice successfully.');
            $status = 'success';
            DB::commit();

            $this->success_notification($title, $message);
        } catch (\Exception $e) {
            DB::rollBack();
            $title = trans('Failed!');
            $this->error_notification($title, $e->getMessage());
        }
        $this->closeModal();
    }

    public function closeModal()
    {
        $this->showDeleteInvoiceModal = false;
        $this->dispatch('refresh-billing-paket');
    }

    public function render()
    {
        return view('livewire.admin.billings.modal.delete-invoice');
    }
}
