<?php

namespace App\Livewire\Admin\Customers;

use App\Models\User;
use Livewire\Component;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use Illuminate\Support\Carbon;
use Livewire\Attributes\Title;
use App\Traits\NotificationTrait;
use Illuminate\Support\Facades\DB;

class CustomerPaketManagement extends Component
{
    use WithPagination, NotificationTrait;
    #[Title('Subscription')]
    //Short by
    public $sortField = 'full_name', $sortDirection = 'asc';
    protected $queryString = ['sortField', 'sortDirection'];

    //Search
    public $search_name = '', $search_with_address = '', $startDateDeadline = '', $endDateDeadline, $perPage = 25;
    public $search_with_paket = '', $search_with_status_customer_paket = '', $search_with_internet_service = '', $selectedServer;
    public $customerPaketSelected = [];
    public $checkAllPaket;


    // Pagination
    public $perPageOptions = [5, 10, 25, 50, 100];

    //dispatch
    public $alert_title, $alert_message;
    public $first_name, $last_name, $address, $email, $phone;

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection == 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    public function updatedSearchNameOrEmail()
    {
        $this->resetPage();
    }

    public function updatedSearchAddress()
    {
        $this->resetPage();
    }

    public function updatedSelectedServer()
    {
        $this->resetPage();
        $this->search_with_paket = '';
    }
    public function updatedSearchWithPaket()
    {
        $this->resetPage();
    }
    public function updatedSearchWithInternetService()
    {
        $this->resetPage();
    }

    public function updatedSearchWithStatusCustomerPaket()
    {
        $this->resetPage();
    }

    /**
     * Alert when user successfully disable or enable
     */
    #[On('user-disable')]
    public function alert(User $model)
    {
        if ($model->disabled) {
            $this->alert_title = trans('user.alert.disable-successfully');
            $this->alert_message = trans('user.alert.user-disable', ['user' => $model->full_name]);
        } else {
            $this->alert_title = trans('user.alert.enable-successfully');
            $this->alert_message = trans('user.alert.user-enable', ['user' => $model->full_name]);
        }
        $this->dispatch('updated');
    }

    /**
     * Export all user
     */
    public function exportUser()
    {
        //  return Excel::download(new UsersExport, 'users.xlsx');
    }


    public function verificationUser(User $user)
    {
        $user->activation();
        $this->success_notification(trans('customer.alert.success'), trans('customer.alert.verification-user-successfully', ['user' => $user->full_name]));
    }


    public function bulkDeleteCustomerPaket()
    {
        $this->dispatch('bulk-delete-customer-paket-modal', customerPaketSelected: $this->customerPaketSelected);
    }

    public function bulkEditCustomerPaket()
    {
        $this->dispatch('bulk-edit-activation-paket-modal', customerPaketSelected: $this->customerPaketSelected);
    }

    #[On('refresh-selected-customer-pakets')]
    public function refreshSelectedCustomerPaket()
    {
        $this->customerPaketSelected = [];
        $this->checkAllPaket = false;
    }

    #[On('notification-user-disable')]
    public function notification($model)
    {
        $this->success_notification(trans('user.alert.user-created'), trans('user.alert.user-created-successfully', ['user' => $model['first_name']]));
    }

    private function get_customers()
    {
        return User::join('user_customers', 'users.id', 'user_customers.user_id')
            ->join('user_addresses', 'users.id', 'user_addresses.user_id')
            ->with('customer_pakets', function ($customer_pakets) {
                //Serach with paket name
                $customer_pakets->when($this->search_with_paket, function ($customer_pakets) {
                    $customer_pakets->where('paket_id', $this->search_with_paket);
                });

                //Search with internet service
                $customer_pakets->when($this->search_with_internet_service, function ($customer_pakets) {
                    $customer_pakets->whereInternetServiceId($this->search_with_internet_service);
                });

                //Search with status
                $customer_pakets->when($this->search_with_status_customer_paket, function ($customer_pakets) {
                    if ($this->search_with_status_customer_paket == "online") {
                        $customer_pakets->where('online', true);
                    } else if ($this->search_with_status_customer_paket == "offline") {
                        $customer_pakets->where('online', false);
                    } else {
                        $customer_pakets->where('status', $this->search_with_status_customer_paket);
                    }
                });

                //Search with server
                $customer_pakets->when($this->selectedServer, function ($customer_pakets) {
                    $customer_pakets->whereHas('paket', function ($pakets) {
                        $pakets->where('mikrotik_id', $this->selectedServer);
                    });
                });

                $customer_pakets->when($this->search_with_address, function ($customer_pakets) {
                    $customer_pakets->whereHas('customer_paket_addresses', function ($customer_paket_addresses) {
                        $customer_paket_addresses->where('address_type', 'installation-address')->where('address', 'like', '%' . $this->search_with_address . '%')
                            ->orWhere('subdistrict', 'like', '%' . $this->search_with_address . '%')
                            ->orWhere('district', 'like', '%' . $this->search_with_address . '%')
                            ->orWhere('city', 'like', '%' . $this->search_with_address . '%')
                            ->orWhere('province', 'like', '%' . $this->search_with_address . '%');
                    });
                });

                $customer_pakets->when($this->startDateDeadline, function ($builder) {
                    if ($this->startDateDeadline && $this->endDateDeadline) {
                        $builder->whereBetween('expired_date', [Carbon::parse($this->startDateDeadline)->startOfDay(), Carbon::parse($this->endDateDeadline)->endOfDay()]);
                    } else {
                        $builder->whereBetween('expired_date', [Carbon::parse($this->startDateDeadline)->startOfDay(), Carbon::now()->endOfDay()]);
                    }
                });
            })
            ->whereHas('customer_pakets', function ($customer_pakets) {
                //Serach with paket name
                $customer_pakets->when($this->search_with_paket, function ($customer_pakets) {
                    $customer_pakets->where('paket_id', $this->search_with_paket);
                });

                //Search with internet service
                $customer_pakets->when($this->search_with_internet_service, function ($customer_pakets) {
                    $customer_pakets->whereInternetServiceId($this->search_with_internet_service);
                });

                //Search with status
                $customer_pakets->when($this->search_with_status_customer_paket, function ($customer_pakets) {
                    if ($this->search_with_status_customer_paket == "online") {
                        $customer_pakets->where('online', true);
                    } else if ($this->search_with_status_customer_paket == "offline") {
                        $customer_pakets->where('online', false);
                    } else {
                        $customer_pakets->where('status', $this->search_with_status_customer_paket);
                    }
                });

                //Search with server
                $customer_pakets->when($this->selectedServer, function ($customer_pakets) {
                    $customer_pakets->whereHas('paket', function ($pakets) {
                        $pakets->where('mikrotik_id', $this->selectedServer);
                    });
                });

                $customer_pakets->when($this->search_with_address, function ($customer_pakets) {
                    $customer_pakets->whereHas('customer_paket_addresses', function ($customer_paket_addresses) {
                        $customer_paket_addresses->where('address_type', 'installation-address')->where('address', 'like', '%' . $this->search_with_address . '%')
                            ->orWhere('subdistrict', 'like', '%' . $this->search_with_address . '%')
                            ->orWhere('district', 'like', '%' . $this->search_with_address . '%')
                            ->orWhere('city', 'like', '%' . $this->search_with_address . '%')
                            ->orWhere('province', 'like', '%' . $this->search_with_address . '%');
                    });
                });

                $customer_pakets->when($this->startDateDeadline, function ($builder) {
                    if ($this->startDateDeadline && $this->endDateDeadline) {
                        $builder->whereBetween('expired_date', [Carbon::parse($this->startDateDeadline)->startOfDay(), Carbon::parse($this->endDateDeadline)->endOfDay()]);
                    } else {
                        $builder->whereBetween('expired_date', [Carbon::parse($this->startDateDeadline)->startOfDay(), Carbon::now()->endOfDay()]);
                    }
                });
            })

            ->when($this->search_name, function ($q) {
                $q->where(function ($builder) {
                    $sql = "CONCAT(users.first_name,' ',COALESCE(users.last_name,''))  like ?";
                    $builder->whereRaw($sql,  "%" . $this->search_name . "%");
                });
            })

            ->select(
                'user_addresses.id as user_address_id',
                'user_addresses.address as address',
                DB::raw("CONCAT(users.first_name,' ',COALESCE(users.last_name,'')) as full_name"),
                'users.*'
            )
            ->orderBy($this->sortField, $this->sortDirection)

            ->paginate($this->perPage);
    }
    #[On('refresh-customer-list')]
    #[On('refresh-customer-paket-list')]
    public function render()
    {
        $customers = $this->get_customers();
        return view('livewire.admin.customers.customer-paket-management', [
            'customers' => $customers
        ]);
    }
}
