<?php

namespace App\Livewire\Admin\WhatsappGateway;

use Livewire\Component;
use App\Traits\StandardPhoneNumber;
use Illuminate\Support\Facades\Validator;
use App\Http\Controllers\API\WhatsappGateway;
use App\Services\WhatsappGateway\GatewayApiService;
use App\Traits\NotificationTrait;

class CompanySettings extends Component
{
    use StandardPhoneNumber, NotificationTrait;
    public $companyName, $slug, $titleMessage = 'ERROR', $errorMessage;
    public $notFound = false;
    public $isError = false;
    public $input = [];

    public function mount()
    {
        try {
            $response = (new GatewayApiService())->getRequest(WhatsappGateway::COMPANY_SETTINGS);

            if ($response['success']) {
                $userCompany = $response['payload']['data']['user_company'];
                $this->companyName = $userCompany['name'] ?? null;
                $this->input['address'] = $userCompany['address'] ?? null;
                $this->input['subdistrict'] = $userCompany['subdistrict'] ?? null;
                $this->input['district'] = $userCompany['district'] ?? null;
                $this->input['city'] = $userCompany['city'] ?? null;
                $this->input['province'] = $userCompany['province'] ?? null;
                $this->input['postal_code'] = $userCompany['postal_code'] ?? null;
                $this->input['country_code'] = $userCompany['country_code'] ?? null;
                $this->input['phone'] = $userCompany['phone'] ?? null;
                $this->input['domain'] = $userCompany['domain'] ?? null;
                $this->slug = $userCompany['slug'] ?? null;
            } else {
                $this->isError = true;
                $this->error_notification('Failed!', $response['message'] ?? 'Unknow error, please contact administrator.');
            }
        } catch (\Exception $e) {
            $this->error_notification('Failed!', $e->getMessage());
            $this->redirect(route('managements.whatsapp_gateway'), navigate: true);
        }
    }


    public function update_company()
    {

        Validator::make(
            $this->input,
            [
                'address' => ['required', 'string', 'max:255'],
                'subdistrict' => ['required', 'string', 'max:255'],
                'district' => ['required', 'string', 'max:255'],
                'city' => ['required', 'string', 'max:255'],
                'province' => ['required', 'string', 'max:255'],
                'postal_code' => ['required', 'numeric', 'min_digits:5', 'max_digits:5'],
                'country_code' => ['required', 'string', 'max:4'],
                'phone' => ['required', 'numeric', 'min_digits:7', 'max_digits:15'],
                'domain' => ['required'],
            ]
        )->validate();

        try {
            $this->input['phone'] = $this->internationalPhoneNumberFormat($this->input['phone']);
            $response = (new GatewayApiService())->updateRequest(WhatsappGateway::COMPANY_SETTINGS, $this->slug, $this->input);

            if ($response['success']) {
                $this->dispatch('company-updated');
                $this->success_notification('Success', $response['payload']['message'] ?? 'Updated successfully');
            } else {
                $this->error_notification('Failed', $response['message'] ?? 'Unknow error, please contact administrator.');
            }
        } catch (\Exception $e) {
            $this->error_notification('Failed', $e->getMessage());
            $this->redirect(route('managements.whatsapp_gateway'), navigate: true);
        }
    }
    public function render()
    {
        return view('livewire.admin.whatsapp-gateway.company-settings');
    }
}
