<?php

namespace App\Livewire\Admin\WhatsappGateway;

use Livewire\Component;
use App\Models\Websystem;
use Livewire\Attributes\On;
use Illuminate\Validation\Rules;
use Illuminate\Support\Facades\Validator;
use App\Http\Controllers\API\WhatsappGateway;
use App\Services\WhatsappGateway\DeviceService;
use App\Services\WhatsappGateway\GatewayApiService;
use App\Services\WhatsappGateway\SubscriptionService;
use Jantinnerezo\LivewireAlert\Facades\LivewireAlert;
use App\Models\WhatsappGateway\WhatsappGatewayGeneral;
use App\Traits\NotificationTrait;

class WhatsappGatewayManagement extends Component
{
    use NotificationTrait;
    public $input = [];
    public $user;
    public $enable = false;
    public $notifAdmin = false;


    public function initialize()
    {
        $company = Websystem::first();
        $this->input['email'] = $company->email;
        $this->input['phone'] = $company->phone;

        Validator::make($this->input, [
            'password' =>  ['required', 'string', Rules\Password::defaults()],
            'email' => ['required', 'string', 'email', 'lowercase', 'max:255'],
            'phone' => ['required', 'numeric', 'min_digits:7', 'max_digits:15',]
        ], [
            'email.required' => 'Please update your company email from General Setting page!',
            'phone.required' => 'Phone number is required, please update from general settings!'
        ])->validate();

        try {
            $response = (new GatewayApiService())->initializeRequest(WhatsappGateway::INIT, $this->input);

            if ($response['success']) {
                $user = $response['payload']['data']['user'];
                $this->dispatch('login-success');
                setEnv('API_CLIENT_SECRET', $user['login_token'] ?? '');
                setEnv('API_CLIENT_MESSAGE', $user['wagateway_token'] ?? '');
                $username = $user['username'];
                setEnv('API_USERNAME', $username);
                $info = 'Syncronyze successfully with ' . $username;
                $this->success_notification('Success', $response['payload']['message']);
                $this->redirect('/managements/whatsapp-gateway', navigate: true);
            } else {
                $this->error_notification('Failed!', $response['error']);
            }
        } catch (\Exception $e) {
            $this->error_notification('Error', $e->getMessage());
        }
    }

    public function updatedEnable($value)
    {
        if ($value) {
            WhatsappGatewayGeneral::first()->enable();
        } else {
            WhatsappGatewayGeneral::first()->disable();
        }
    }

    public function updatedNotifAdmin($value)
    {
        if ($value) {
            WhatsappGatewayGeneral::first()->sendWaAdminEnable();
        } else {
            WhatsappGatewayGeneral::first()->sendWaAdminDisable();
        }
    }

    public function updatedInputWhatsappNumberBoot($value)
    {
        $this->input['number'] = $value;

        try {
            $response = (new GatewayApiService())->addRequest(WhatsappGateway::NOTIFICATION_DEVICES,  $this->input);
            //dd($response);
            if ($response['result']['success']) {
                if (empty($value)) {
                    WhatsappGatewayGeneral::first()->forceFill(['whatsapp_number_boot' => null])->save();
                } else {
                    WhatsappGatewayGeneral::first()->forceFill(['whatsapp_number_boot' => $value])->save();
                }

                $this->disableWaGateway();
                $this->dispatch('boot-number-updated');
            } else {
                $this->dispatch('boot-number-failed-updated');
            }
        } catch (\Exception $e) {
            $this->dispatch('boot-number-failed-updated');
        }
        // $this->dispatch('boot-number-failed-updated');
    }

    public function updatedInputWhatsappNumberNotification($value)
    {
        if (empty($value)) {
            WhatsappGatewayGeneral::first()->forceFill(['whatsapp_number_notification' => null])->save();
        } else {
            WhatsappGatewayGeneral::first()->forceFill(['whatsapp_number_notification' => $value])->save();
        }
        $this->disableWaGateway();

        $this->dispatch('notification-number-updated');
    }

    private function disableWaGateway()
    {
        if (is_null(WhatsappGatewayGeneral::first()->whatsapp_number_notification) && is_null(WhatsappGatewayGeneral::first()->whatsapp_number_boot)) {
            WhatsappGatewayGeneral::first()->disable();
            WhatsappGatewayGeneral::first()->sendWaAdminDisable();
        }
    }

    public function updatedInputRemainingDay($value)
    {
        Validator::make($this->input, [
            'remaining_day' => ['required', 'numeric', 'max:15']
        ])->validate();
        WhatsappGatewayGeneral::first()->forceFill(['remaining_day' => $value])->save();
        $this->dispatch('remaining-day-updated');
    }

    public function updatedInputScheduleTime($value)
    {
        Validator::make($this->input, [
            'schedule_time' => ['required'],
        ])->validate();

        setEnv('WA_REMINDER_TIME', $value);
        $this->dispatch('schedule-time-updated');
    }


    #[On('refresh-subscription-management')]
    public function render(SubscriptionService $subscriptionService, DeviceService $deviceService)
    {
        $notifications = [];
        $devices = [];
        $subscription = [];

        try {
            $siteOnline = true;
            $response = (new GatewayApiService())->getRequest(WhatsappGateway::SUBSCRIPTION);


            if ($response['success']) {
                $isLogin = true;
                $data = $response['payload']['data'];
                $notifications = $data['notifications'] ?? [];

                $devices = collect($data['devices'])->where('status', 'Connected');
                $this->user = $data['user'];
                $subscription = $data['subscription'] ?? [];
                $whatsappGateway = WhatsappGatewayGeneral::first();
                $this->notifAdmin = $whatsappGateway->send_wa_admin ? true : false;
                $this->enable =  $whatsappGateway->disabled ? false : true;
                $this->input = array_merge([
                    'schedule_time' => env('WA_REMINDER_TIME'),
                    'country_code_number' => $whatsappGateway->country_code
                ], $whatsappGateway->withoutRelations()->toArray());
            } else {
                $isLogin = false;
            }
        } catch (\Exception $e) {
            $isLogin = false;
            // $this->notification('Error', $e->getMessage(), 'error');
            $siteOnline = false;
        }

        return view('livewire.admin.whatsapp-gateway.whatsapp-gateway-management', compact('isLogin', 'siteOnline', 'devices', 'subscription', 'notifications'));
    }
}
