<?php

namespace App\Models\Customers;

use App\Models\Websystem;
use Spatie\Sluggable\HasSlug;
use Illuminate\Support\Carbon;
use App\Models\Billings\Invoice;
use Spatie\Sluggable\SlugOptions;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Prunable;
use Illuminate\Database\Eloquent\SoftDeletes;
use Dyrynda\Database\Support\CascadeSoftDeletes;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;

class CustomerPaket extends Model
{
    use HasFactory, SoftDeletes, CascadeSoftDeletes;
    use HasSlug;
    use Prunable;

    protected $cascadeDeletes = ['customer_ppp_pakets', 'customer_static_pakets', 'customer_paket_addresses', 'invoices', 'mikrotik_client_histories'];
    protected $dates = ['deleted_at'];
    protected $fetchMethod = 'get'; // get, cursor, lazy or chunk
    public $guarded = [];
    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()
            ->generateSlugsFrom(['user_id', 'paket_id'])
            ->saveSlugsTo('slug')
            ->usingSuffixGenerator(
                fn(string $slug, int $iteration) => bin2hex(random_bytes(6))
            )
            ->doNotGenerateSlugsOnUpdate();
    }
    public function getRouteKeyName(): string
    {
        return 'slug';
    }

    public function prunable()
    {
        //return static::where('deleted_at', '>=', now()->subDays(30));
        return static::where('deleted_at', '<=', Carbon::now()->subDays(30));
    }

    function customer_ppp_paket(): HasOne
    {
        return $this->hasOne(\App\Models\Customers\CustomerPppPaket::class);
    }

    function customer_ppp_pakets(): HasMany
    {
        return $this->hasMany(\App\Models\Customers\CustomerPppPaket::class);
    }

    public function customer_static_paket(): HasOne
    {
        return $this->hasOne(\App\Models\Customers\CustomerStaticPaket::class);
    }

    public function customer_static_pakets(): HasMany
    {
        return $this->hasMany(\App\Models\Customers\CustomerStaticPaket::class);
    }

    function customer_paket_address(): HasOne
    {
        return $this->hasOne(\App\Models\Customers\CustomerPaketAddress::class);
    }

    function customer_paket_addresses(): HasMany
    {
        return $this->hasMany(\App\Models\Customers\CustomerPaketAddress::class);
    }

    public function customer_billing_address(): HasOne
    {
        return $this->hasOne(\App\Models\Customers\CustomerPaketAddress::class, 'customer_paket_id', 'id')->whereAddressType('billing-address');
    }

    public function customer_installation_address(): HasOne
    {
        return $this->hasOne(\App\Models\Customers\CustomerPaketAddress::class, 'customer_paket_id', 'id')->whereAddressType('installation-address');
    }

    public function internet_service(): BelongsTo
    {
        return $this->belongsTo(\App\Models\Pakets\InternetService::class);
    }

    public function paket(): BelongsTo
    {
        return $this->belongsTo(\App\Models\Pakets\Paket::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(\App\Models\User::class);
    }

    public function invoices(): HasMany
    {
        return $this->hasMany(Invoice::class);
    }

    public function invoice(): HasOne
    {
        return $this->hasOne(Invoice::class);
    }

    public function unpayment_invoices(): HasMany
    {
        return $this->hasMany(Invoice::class)->where('status', '!=', 'paid');
    }

    public function mikrotik()
    {
        return $this->paket->mikrotik();
    }

    public function activation($activationDate = null)
    {
        $this->activation_date = $activationDate ? $activationDate : now();
        $this->status = 'active';
        return $this->save();
    }

    public function cancel()
    {
        $this->auto_renew = false;
        $this->status = 'cancelled';
        return $this->save();
    }

    public function suspend()
    {
        $this->status = 'suspended';
        return $this->save();
    }

    public function resume()
    {
        if ($this->status === 'suspended') {
            $this->status = 'active';
            return $this->save();
        }
        return false;
    }

    public function checkSubscriptionStatus()
    {
        $expiredDate = $this->paylater_date ? (Carbon::parse($this->expired_date)->gt(Carbon::parse($this->paylater_date)) ? $this->expired_date : $this->paylater_date) : $this->expired_date;
        if (Carbon::parse($expiredDate)->isPast()) {
            $this->status = 'expired';
            $this->save();
        }
        return $this->status;
    }

    public function setSubscriptionStatus()
    {
        $expiredDate = $this->paylater_date ? (Carbon::parse($this->expired_date)->gt(Carbon::parse($this->paylater_date)) ? $this->expired_date : $this->paylater_date) : $this->expired_date;
        if (Carbon::parse($this->start_date)->lte(Carbon::parse($this->activation_date))) {
            $this->status = 'pending';
            if ($this->isPpp()) {
                $this->customer_ppp_paket->secret_id = null;
            } elseif ($this->isIpStatic()) {
                $this->customer_static_paket->simpleque_id = null;
            }
        } else if (Carbon::parse($expiredDate)->isPast()) {
            $this->status = 'expired';
        } else {
            $this->status = 'active';
        }
        $this->save();
        return $this->status;
    }

    // public function setActive()
    // {
    //    $this->status = 'active';
    //     $this->save();
    // }
    public function isActive()
    {
        return $this->status === 'active';
    }

    public function needsBilling($nextBilledAt)
    {
        if (!$this->next_billed_at) {
            return true;
        }

        $customerPaketInvoice = $this->invoices()
            ->whereDate('start_periode', Carbon::parse($this->expired_date)->format('Y-m-d'))
            ->first();

        //$nextBilledAt =  Carbon::parse($this->next_billed_at);
        return $nextBilledAt->isPast() && !$customerPaketInvoice;
    }

    public function getRenewalPeriod()
    {
        switch ($this->renewal_period) {
            case 'monthly':
                return '1 month';
            case 'bimonthly':
                return '2 month';
            case 'quarterly':
                return '3 months';
            case 'semi-annually':
                return '6 months';
            case 'annually':
                return '1 year';
            default:
                return '1 month';
        }
    }

    public function mikrotik_client_histories(): HasMany
    {
        return $this->hasMany(\App\Models\Servers\MikrotikClientHistory::class);
    }

    public function isPpp()
    {
        return $this->internet_service->value === 'ppp';
    }

    public function isIpStatic()
    {
        return $this->internet_service->value === 'ip_static';
    }

    protected static function boot()
    {
        parent::boot();
        static::restoring(function ($customer_paket) {
            $customer_paket->customer_static_paket()->withTrashed()->restore();
            $customer_paket->customer_ppp_paket()->withTrashed()->restore();
            $customer_paket->customer_paket_addresses()->withTrashed()->restore();
            $customer_paket->invoices()->withTrashed()->restore();
        });

        /*static::updating(function ($customer_paket) {
            $this->customer_paket_status_latest->forceFill([
                'status' => $this->status,
                'activation_date' => $this->activation_date,
                'start_date' => $this->start_date,
                'expired_date' => $this->expired_date,
                'paylater_date' => $this->paylater_date,
                'last_billed_at' => $this->last_billed_at
            ])->save();
        });*/
    }

    public function payments(): HasManyThrough
    {
        return $this->hasManyThrough(\App\Models\Billings\Payment::class, Invoice::class);
    }

    public function partial_payments(): HasManyThrough
    {
        return $this->hasManyThrough(\App\Models\Billings\Payment::class, Invoice::class)->where('invoices.status', '!=', 'paid');
    }

    //public function customer_paket_status_latest(): HasOne
   // {
    //    return $this->hasOne(\App\Models\Customers\CustomerPaketStatusLatest::class);
   // }

    public function getFullCustomerBillingAddressAttribute()
    {
        $billingAddress = $this->customer_billing_address()->first();
        $address = $billingAddress->address ? $billingAddress->address . ', ' : '';
        $subdistrict = $billingAddress->subdistrict ? $billingAddress->subdistrict . ', ' : '';
        $district = $billingAddress->district ? $billingAddress->district . ', ' : '';
        $city = $billingAddress->city ? $billingAddress->city . ', ' : '';
        $province = $billingAddress->province ? $billingAddress->province . ', ' : '';
        $country = $billingAddress->country ? $billingAddress->country . ', ' : '';
       return $address . $subdistrict . $district . $city . $province . $country;
    }

    public function getFullCustomerInstallationAddressAttribute()
    {
        $installationAddress = $this->customer_installation_address()->first();
        $address = $installationAddress->address ? $installationAddress->address . ', ' : '';
        $subdistrict = $installationAddress->subdistrict ? $installationAddress->subdistrict . ', ' : '';
        $district = $installationAddress->district ? $installationAddress->district . ', ' : '';
        $city = $installationAddress->city ? $installationAddress->city . ', ' : '';
        $province = $installationAddress->province ? $installationAddress->province . ', ' : '';
        $country = $installationAddress->country ? $installationAddress->country . ', ' : '';
       return $address . $subdistrict . $district . $city . $province . $country;
    }

    public function getNetPriceAttribute()
    {
        $config = Websystem::first();
        $netPrice =  $this->price - $this->discount;
        return $config->enable_tax ? $netPrice + ($netPrice * ($config->tax_rate / 100)) : $netPrice;
    }
}
