<?php

namespace App\Observers;

use App\Traits\WebSystemTrait;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;
use App\Services\GeneralLogServices;
use App\Services\CustomerPaketService;
use App\Models\Customers\CustomerPaket;
use App\Services\Mikrotiks\MikrotikPppService;
use App\Services\Mikrotiks\MikrotikIpStaticService;
use App\Jobs\Notifications\SendNotificationSubscriptionStatusChangeJob;
use App\Jobs\Notifications\SendNotificationCustomerPaketConnectionStatusJob;

class CustomerPaketObserver
{
    use WebSystemTrait;
    private $generalLogServices, $customerPaketService;
    public function __construct(CustomerPaketService $customerPaketService = null, GeneralLogServices $generalLogServices = null)
    {
        $this->customerPaketService = $customerPaketService ?? new CustomerPaketService();
        $this->generalLogServices = $generalLogServices ?? new GeneralLogServices;
    }

    /**
     * Handle the CustomerPaket "created" event.
     */
    public function created(CustomerPaket $customerPaket): void {}

    /**
     * Handle the CustomerPaket "updated" event.
     */
    public function updated(CustomerPaket $customerPaket): void
    {
        if ($customerPaket->isDirty('status')) {
            $mikrotik = $customerPaket->paket->mikrotik;
            switch ($customerPaket->status) {
                case 'expired':
                    dispatch(new SendNotificationSubscriptionStatusChangeJob($customerPaket, 'sendIsolirNotification', $this->generalLogServices::NOTIFIACTION_EXPIRED))->onQueue('default');
                    // $this->customerPaketService->send_whatsapp_notification_to_installation_address($customerPaket, 'sendIsolirNotification', $this->generalLogServices::NOTIFIACTION_EXPIRED);
                    switch ($customerPaket->internet_service->value) {
                        case 'ppp':
                            $customerPppPaket = $customerPaket->customer_ppp_paket;
                            $profileIsolir = $mikrotik->auto_isolir->profile_id;
                            (new MikrotikPppService())->updateProfileSecret($mikrotik, $customerPppPaket, $profileIsolir);
                            break;
                        case 'ip_static':
                            // $customerStaticPaket = $customerPaket->customer_static_paket;
                            //Isolir ip static in development
                            break;
                        default:
                            break;
                    }

                    break;
                case 'cancelled':
                    dispatch(new SendNotificationSubscriptionStatusChangeJob($customerPaket, 'sendCancelledCustomerPaketNotification'))->onQueue('default');
                    //$this->customerPaketService->send_whatsapp_notification_to_installation_address($customerPaket, 'sendCancelledCustomerPaketNotification', null);
                    switch ($customerPaket->internet_service->value) {
                        case 'ppp':
                            $customerPppPaket = $customerPaket->customer_ppp_paket;
                            (new MikrotikPppService())->deleteSecret($mikrotik, $customerPppPaket);
                            break;
                        case 'ip_static':
                            $customerStaticPaket = $customerPaket->customer_static_paket;
                            (new MikrotikIpStaticService())->deleteIpStaticPaket($mikrotik, $customerStaticPaket->ip_address);
                            break;
                        default:
                            break;
                    }

                    break;
                case 'suspended':
                    dispatch(new SendNotificationSubscriptionStatusChangeJob($customerPaket, 'sendSuspendedCustomerPaketNotification'))->onQueue('default');
                    //$this->customerPaketService->send_whatsapp_notification_to_installation_address($customerPaket, 'sendSuspendedCustomerPaketNotification', null);
                    switch ($customerPaket->internet_service->value) {
                        case 'ppp':
                            $customerPppPaket = $customerPaket->customer_ppp_paket;
                            (new MikrotikPppService())->disableSecret($mikrotik, $customerPppPaket, 'true');
                            break;
                        case 'ip_static':
                            $customerStaticPaket = $customerPaket->customer_static_paket;
                            (new MikrotikIpStaticService())->disableIpFromArp($mikrotik, $customerStaticPaket->arp_id, 'true');
                            break;
                        default:
                            break;
                    }
                    break;
                case 'active':
                    $lastStatus = $customerPaket->getOriginal('status');
                    if ($lastStatus != 'pending') {
                        if ($lastStatus != 'cancelled') {
                            dispatch(new SendNotificationSubscriptionStatusChangeJob($customerPaket, 'sendRestoreCustomerPaketNotification'))->onQueue('default');
                        }
                    }
                    //$this->customerPaketService->send_whatsapp_notification_to_installation_address($customerPaket, 'sendRestoreCustomerPaketNotification', null);
                    $comment = $this->websystem_comment_unpayment() . '_' . Carbon::parse($customerPaket->expired_date)->format('d_m_Y');
                    if ($lastStatus === 'expired') {
                        switch ($customerPaket->internet_service->value) {
                            case 'ppp':
                                $customerPppPaket = $customerPaket->customer_ppp_paket;
                                $userProfile = $customerPppPaket->paket->paket_profile->profile_name;
                                (new MikrotikPppService)->updateProfileSecret($mikrotik, $customerPppPaket, $userProfile, $comment);
                                break;
                            case 'ip_static':
                                $ipAddress = $customerPaket->customer_static_paket->ip_address;
                                (new MikrotikIpStaticService)->updateCommentSimpleQueue($mikrotik, $ipAddress, $comment);
                                (new MikrotikIpStaticService)->updateCommentArp($mikrotik, $ipAddress, $comment);
                                (new MikrotikIpStaticService)->deleteIpFromAddressList($mikrotik, $ipAddress, $mikrotik->auto_isolir->address_list_isolir);
                                break;
                            default:
                                break;
                        }
                    } elseif ($lastStatus === 'suspended') {
                        switch ($customerPaket->internet_service->value) {
                            case 'ppp':
                                $customerPppPaket = $customerPaket->customer_ppp_paket;
                                $userProfile = $customerPppPaket->paket->paket_profile->profile_name;
                                (new MikrotikPppService)->disableSecret($mikrotik, $customerPppPaket, 'false');
                                (new MikrotikPppService)->updateProfileSecret($mikrotik, $customerPppPaket, $userProfile);
                                break;
                            case 'ip_static':
                                //
                                break;
                            default:
                                break;
                        }
                    } elseif ($lastStatus === 'pending') {

                        switch ($customerPaket->internet_service->value) {
                            case 'ppp':
                                $customerPppPaket = $customerPaket->customer_ppp_paket;
                                (new MikrotikPppService)->updateCommentSecret($mikrotik, $customerPppPaket, $comment);
                                break;
                            case 'ip_static':
                                //
                                break;
                            default:
                                break;
                        }
                    }

                    break;

                default:
                    # code...
                    break;
            }
        }

        if ($customerPaket->isDirty('online')) {
            dispatch(new SendNotificationCustomerPaketConnectionStatusJob($customerPaket, $customerPaket->online))->onQueue('default')->delay(120);
        }
    }

    public function updating(CustomerPaket $customerPaket)
    {
        /*

        $customerPaketStatusLatest = $customerPaket->customer_paket_status_latest;
        $customerPaketStatusLatest->forceFill([
            'status' => $customerPaket->isDirty('status') ? $customerPaket->getOriginal('status') : $customerPaketStatusLatest->status,
            'activation_date' => $customerPaket->isDirty('activation_date') ? $customerPaket->getOriginal('activation_date') : $customerPaketStatusLatest->activation_date,
            'start_date' => $customerPaket->isDirty('start_date') ? $customerPaket->getOriginal('start_date') : $customerPaketStatusLatest->start_date,
            'expired_date' => $customerPaket->isDirty('expired_date') ? $customerPaket->getOriginal('expired_date') : $customerPaketStatusLatest->expired_date,
            'paylater_date' => $customerPaket->isDirty('paylater_date') ? $customerPaket->getOriginal('paylater_date') : $customerPaketStatusLatest->paylater_date,
            'next_billed_at' => $customerPaket->isDirty('next_billed_at') ? $customerPaket->getOriginal('next_billed_at') : $customerPaketStatusLatest->next_billed_at,
        ])->save();

        */
    }

    /**
     * Handle the CustomerPaket "deleted" event.
     */
    public function deleted(CustomerPaket $customerPaket): void
    {
        // $customerPaket->customer_paket_status_latest()->withTrashed()->delete();
    }

    /**
     * Handle the CustomerPaket "restored" event.
     */
    public function restored(CustomerPaket $customerPaket): void
    {
        //
    }

    /**
     * Handle the CustomerPaket "force deleted" event.
     */
    public function forceDeleted(CustomerPaket $customerPaket): void
    {
        // $customerPaket->customer_paket_status_latest()->withTrashed()->forceDeleted();
    }
}
