<?php

namespace App\Services;

use App\Models\User;
use App\Models\Websystem;
use App\Models\Pakets\Paket;
use App\Models\Pakets\PppType;
use App\Traits\WebSystemTrait;
use Illuminate\Support\Carbon;
use App\Traits\CustomerPaketTrait;
use App\Models\Customers\AutoIsolir;
use Illuminate\Support\Facades\Auth;
use App\Models\Customers\CustomerPaket;
use App\Services\Billings\BillingService;
use App\Models\Customers\CustomerPppPaket;
use App\Services\Billings\DeadlineService;
use App\Models\Customers\CustomerStaticPaket;
use App\Services\Mikrotiks\MikrotikPppService;
use App\Services\Mikrotiks\MikrotikIpStaticService;
use App\Jobs\Billings\ProcessSubscriptionBillingJob;
use App\Livewire\Actions\Billings\InvoiceAction;
use App\Livewire\Actions\Customers\CustomerPaketAction;
use App\Services\WhatsappGateway\WhatsappNotificationService;
use App\Livewire\Actions\Customers\CustomerPaketAddressAction;

class CustomerPaketService
{
    use WebSystemTrait, CustomerPaketTrait;
    protected $pricingService, $deadlineService;
    private $mikrotikPppService;

    private $generalLogServices, $whatsappNotificationService;


    public function __construct(
        DeadlineService $deadlineService = null,
        MikrotikPppService $mikrotikPppService = null,
        GeneralLogServices $generalLogServices = null,
        WhatsappNotificationService $whatsappNotificationService = null,
        //BillingService $billingService = null
    ) {
        $this->deadlineService = $deadlineService ?? new DeadlineService();
        $this->mikrotikPppService = $mikrotikPppService ?? new MikrotikPppService();
        $this->generalLogServices = $generalLogServices ?? new GeneralLogServices;
        $this->whatsappNotificationService = $whatsappNotificationService ?? new WhatsappNotificationService;
        // $this->billingService = $billingService ?? new BillingService;
    }

    /**
     * Summary of addCustomerPaket
     * @param \App\Models\User $user
     * @param mixed $input
     * @return CustomerPaket
     */
    public function addCustomerPaket(User $user, $input)
    {
        $paket = Paket::find($input['selectedPaket']);
        //Create Customer Paket
        $customerPaket = (new CustomerPaketAction())->createCustomerPaket($user, $paket, $input);

        if ($this->isPrabayar()) {
            //Create invoice first time

            if ($paket->trial_days <= 0) {
                $invoice = (new BillingService())->generateInvoice($customerPaket);
                /* $invoice->forceFill([
                    'due_date' => Carbon::parse($invoice->due_date)->addDays($trialDays),
                    'start_periode' => Carbon::parse($invoice->start_periode)->addDays($trialDays),
                    'end_periode' =>  Carbon::parse($invoice->end_periode)->addDays($trialDays),
                ])->save();*/
            }
        }
        return $customerPaket;
    }

    public function createUserOnMikrotik(CustomerPaket $customerPaket)
    {
        $mikrotik = $customerPaket->paket->mikrotik;
        if ($this->autoIsolirDriverMikrotik()) {
            $comment = $this->commentMikrotik($customerPaket->expired_date);
        } else {
            $comment = Carbon::parse($customerPaket->expired_date)->format('d-m-Y');
        }

        try {
            if ($customerPaket->isPpp()) {
                //PPP Service
                //return $this->create_ppp_secret($mikrotik, $customerPaket, $comment, 'false');
                $disabled = $customerPaket->status === 'suspended' ? 'true' : 'false';
                $pppSecret = (new MikrotikPppService())->createSecret($mikrotik, $customerPaket, $disabled, $comment);
                if ($pppSecret['success']) {
                    $secretID = $pppSecret['secret_id'];
                    (new CustomerPaketAction())->update_secret_id($customerPaket, $secretID);
                    //$customerPaket->activation();
                    return [
                        'success' => true,
                        'message' => trans('customer.paket.alert.customer-paket-activation-detail', ['customer' => $customerPaket->user->full_name, 'paket' => $customerPaket->paket->name])
                    ];
                }
                return $pppSecret;
            } else {
                //IP Static
                // return $this->create_simpleque_ip_static($mikrotik, $customerPaket, $comment);
                $paketProfile = $customerPaket->paket->paket_profile;
                $ipStaticPaket = $customerPaket->customer_static_paket;
                $ip_static = $ipStaticPaket->ip_address;
                $mac_address = $ipStaticPaket->mac_address;
                $interface = $ipStaticPaket->interface;
                $simpleQueueName = $customerPaket->user->first_name . $ipStaticPaket->id;

                try {
                    $createLimitasi = (new MikrotikIpStaticService())->createSimpleQueue($mikrotik, $simpleQueueName, $ip_static, $paketProfile, $comment);
                    if (isset($createLimitasi['after']['message'])) {
                        return [
                            'success' => false,
                            'message' => $createLimitasi['after']['message']
                        ];
                    } else {
                        $simpleque_id = $createLimitasi['after']['ret'];
                    }

                    $createArp = (new MikrotikIpStaticService())->addIpToArp($mikrotik, $ip_static, $mac_address, $interface, $comment);

                    if (isset($createArp['after']['message'])) {
                        (new MikrotikIpStaticService())->deleteIpFromSimpleQueue($mikrotik, $simpleque_id);
                        return [
                            'success' => false,
                            'message' => $createArp['after']['message']
                        ];
                    } else {
                        $arp_id = $createArp['after']['ret'];
                        (new CustomerPaketAction())->update_ip_static_paket_id($ipStaticPaket, $simpleque_id, $arp_id, $simpleQueueName);
                        //$customerPaket->activation();
                        return [
                            'success' => true,
                            'message' => trans('customer.paket.alert.customer-paket-activation-detail', ['customer' => $customerPaket->user->full_name, 'paket' => $customerPaket->paket->name])
                        ];
                    }
                } catch (\Exception $e) {
                    return [
                        'success' => false,
                        'message' => $e->getMessage()
                    ];
                }
            }
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    public function activationCustomerPaket(CustomerPaket $customerPaket, $startPaymentDate = null)
    {
        $paket = $customerPaket->paket;
        $trialDays = $paket->trial_days;
        if ($customerPaket->status === 'cancelled') (new BillingService())->generateInvoice($customerPaket);

        $activationDate = Carbon::now();
        $renewalPeriod = $this->getRenewalPeriod($customerPaket->renewal_period);
        $isSetStartPaymentDate = is_null($startPaymentDate) || empty($startPaymentDate) ? false : true;
        $mikrotikAutoIsolir = $customerPaket->mikrotik->auto_isolir;
        if ($mikrotikAutoIsolir->activation_date) {
            if ($this->isPrabayar()) {
                $expiredDate = $isSetStartPaymentDate ? Carbon::parse($startPaymentDate) : Carbon::now()->addDays($trialDays);
                $nextBilledAt = Carbon::parse($expiredDate)->add($renewalPeriod)->subDays($this->different_day_create_billing());
                $startDate = $expiredDate;
            } else {
                $expiredDate = $isSetStartPaymentDate ? Carbon::parse($startPaymentDate) : Carbon::now()->add($renewalPeriod);
                $intervalInvoiceDay = $this->different_day_create_billing();
                $nextBilledAt = Carbon::parse($expiredDate)->subDays($intervalInvoiceDay);
                $startDate = Carbon::parse($expiredDate)->sub($renewalPeriod);
            }
        } else {
            if ($this->isPrabayar()) {
                $expiredDate = $isSetStartPaymentDate ? Carbon::parse($startPaymentDate) : Carbon::parse($mikrotikAutoIsolir->due_date);
                $nextBilledAt = Carbon::parse($expiredDate)->add($renewalPeriod)->subDays($this->different_day_create_billing());
                $startDate = $expiredDate;
            } else {

                $expiredDate = $isSetStartPaymentDate ? Carbon::parse($startPaymentDate) : Carbon::parse($mikrotikAutoIsolir->due_date)->add($renewalPeriod);
                $intervalInvoiceDay = $this->different_day_create_billing();
                $nextBilledAt = Carbon::parse($expiredDate)->subDays($intervalInvoiceDay);
                $startDate = Carbon::parse($expiredDate)->sub($renewalPeriod);
            }
        }

        $createUserOnMikrotik = $this->createUserOnMikrotik($customerPaket);

        if ($createUserOnMikrotik['success']) {
            $customerPaket->forceFill([
                'start_date' => $startDate,
                'expired_date' => $expiredDate,
                'activation_date' => $activationDate,
                'next_billed_at' => $nextBilledAt,
                'status' => 'active'
            ])->save();

            if ($trialDays > 0 && $this->isPrabayar()) {
                (new BillingService())->generateInvoice($customerPaket);
            }
        }

        return $createUserOnMikrotik;
    }

    public function editActivationDate(CustomerPaket $customerPaket, $activationDate, $startDate, $expiredDate, $nextBilledAt)
    {
        if ($activationDate) {
            $customerPaket->forceFill([
                'activation_date' => $activationDate,
            ])->save();
        }

        if ($startDate) {
            $customerPaket->forceFill([
                'start_date' => $startDate,
                'expired_date' => $expiredDate,
                'next_billed_at' => $nextBilledAt
            ])->save();

            $this->synchronizationInvoice($customerPaket, $startDate, $expiredDate, $nextBilledAt);
        }
    }

    private function synchronizationInvoice(CustomerPaket $customerPaket, $startDate, $expiredDate, $nextBilledAt)
    {
        $renewalPeriod = $customerPaket->getRenewalPeriod();
        if ($this->isPrabayar()) {
            $invoice = $customerPaket->invoices()->where('periode', Carbon::parse($expiredDate)->startOfMonth())->first();
        } else {
            $invoice = $customerPaket->invoices()->where('periode', Carbon::parse($startDate)->startOfMonth())->first();
        }

        if ($invoice) {
            $dayActivation = (int)Carbon::parse($startDate)->format('d');
            $startPeriode = Carbon::parse($invoice->periode)->setDay($dayActivation);
            $endPeriode = Carbon::parse($startPeriode)->add($renewalPeriod);

            if ($invoice->status != 'paid') {
                if ($nextBilledAt->isPast()) {
                    $invoice->forceFill([
                        'start_periode' => $startPeriode,
                        'end_periode' => $endPeriode,
                        'due_date' => $this->isPrabayar() ?  $startPeriode : $endPeriode,
                    ])->save();
                } else {
                    //$invoice->forceDelete();
                    (new InvoiceAction)->delete_billing($invoice);
                }
            } else {
                $customerPaket->forceFill([
                    'start_date' => $startPeriode,
                    'expired_date' => $endPeriode,
                    'next_billed_at' => Carbon::parse($endPeriode)->subDays((int)$this->different_day_create_billing())
                ])->save();
            }
        } else {
            if (Carbon::parse($customerPaket->next_billed_at)->isPast()) dispatch(new ProcessSubscriptionBillingJob($customerPaket))->onQueue('default');
        }
    }

    /**
     * Summary of update_ppp_paket
     * @param \App\Models\Customers\CustomerPppPaket $customerPppPaket
     * @param array $input
     * @return array{message: string, success: bool|array{success: bool}}
     */
    public function update_ppp_paket(CustomerPppPaket $customerPppPaket, array $input)
    {
        try {
            if (!is_null($customerPppPaket->customer_paket->activation_date)) {
                // dd($customerPppPaket->customer_paket->activation_date);
                $pppTypeName = PppType::whereId($input['selectedPppService'] ?? $customerPppPaket->ppp_type_id)->first()->name;
                $response = (new MikrotikPppService())->updateUsernameAndPasswordSecret(
                    $customerPppPaket->customer_paket->paket->mikrotik,
                    $customerPppPaket,
                    $input['username'],
                    $input['password_ppp'],
                    $pppTypeName
                );
            }
            if ($response['success']) {
                (new CustomerPaketAction())->updatePppPaket($customerPppPaket, $input);
            }
            return $response;
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Summary of update_ip_static
     * @param \App\Models\Customers\CustomerStaticPaket $customerStaticPaket
     * @param array $input
     * @return array{message: string, success: bool|array{success: bool}}
     */
    public function update_ip_static(CustomerStaticPaket $customerStaticPaket, array $input)
    {
        try {
            $paket = $customerStaticPaket->customer_paket->paket;
            $mikrotik = $paket->mikrotik;
            if (!is_null($customerStaticPaket->customer_paket->activation_date)) {
                (new MikrotikIpStaticService())->updateSimpleQueue($mikrotik, $customerStaticPaket->simpleque_id, $paket->paket_profile, $customerStaticPaket->ip_address);
            }
            (new CustomerPaketAction())->updateStaticPaket($customerStaticPaket, $input);
            return  [
                'success' => true,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Summary of update_customer_paket
     * @param \App\Models\Customers\CustomerPaket $customerPaket
     * @param \App\Models\Pakets\Paket $paket
     * @return array{message: string, success: bool|array{success: bool}}
     */
    public function update_customer_paket(CustomerPaket $customerPaket, Paket $paket)
    {
        try {
            if (!is_null($customerPaket->activation_date)) {
                if ($customerPaket->internet_service->value == 'ppp') {
                    $response = (new MikrotikPppService())->updateProfileSecret(
                        $customerPaket->paket->mikrotik,
                        $customerPaket->customer_ppp_paket,
                        $paket->paket_profile->profile_name
                    );
                    if (!$response['success']) return $response;
                } else if ($customerPaket->internet_service->value == 'ip_static') {
                    $response = (new MikrotikIpStaticService())->updateSimpleQueue(
                        $customerPaket->paket->mikrotik,
                        $customerPaket->customer_static_paket->simpleque_id,
                        $paket->paket_profile->profile_name,
                        $customerPaket->customer_static_paket->ip_address
                    );
                    if (!$response['success']) return $response;
                }
            }
            (new CustomerPaketAction())->updatePaketCustomerPaket($customerPaket, $paket);
            return  [
                'success' => true,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Summary of deleteCustomerPaket
     * @param \App\Models\Customers\CustomerPaket $customerPaket
     * @return void
     */
    public function deleteCustomerPaket(CustomerPaket $customerPaket, $deleteOnMikrotik = false)
    {
        if ($deleteOnMikrotik) {
            $deleteSecretOnMikrotik = $this->delete_user_on_mikrotik($customerPaket);
            if ($deleteSecretOnMikrotik['success']) {
                $customerPaket->customer_ppp_paket->forceFill(['secret_id' => null])->save();
                // $customerPaket->forceFill(['status' => 'pending'])->save();
            }
        } else {
            $this->disableCustomerPaketOnMikrotik($customerPaket, 'true');
        }

        (new CustomerPaketAction())->delete_paket($customerPaket);
    }

    /**
     * Summary of delete_user_on_mikrotik
     * @param mixed $customerPaket
     *
     */
    public function delete_user_on_mikrotik($customerPaket)
    {
        $mikrotik = $customerPaket->paket->mikrotik;

        if ($customerPaket->isPpp()) {
            $customerPppPaket = $customerPaket->customer_ppp_paket()->withTrashed()->first();
            return (new MikrotikPppService())->deleteSecret($mikrotik, $customerPppPaket);
        } else if ($customerPaket->isIpStatic()) {
            $customerStaticPaket = $customerPaket->customer_static_paket()->withTrashed()->first();
            $arp =  (new MikrotikIpStaticService())->deleteIpStaticPaket($mikrotik, $customerStaticPaket->ip_address);
            if (isset($arp['after']['message'])) return ['success' => false, 'message' => $arp['after']['message']];
            return ['success' => true];
        }
    }

    public function updateStatusCustomerPaket(CustomerPaket $customerPaket, $status)
    {
        switch ($status) {
            case 'active':
                //return $this->restore_customer_paket($customerPaket);
                (new CustomerPaketAction())->update_status_customer_paket($customerPaket, 'active');
                break;
            case 'cancelled':
                // return $this->cancelled_customer_paket($customerPaket);
                $customerPaket->forceFill([
                    'start_date' => null,
                    'expired_date' => null,
                    'next_billed_at' => null,
                    'activation_date' => null,
                    'status' => 'cancelled'
                ])->save();

                //Delete unpayment invoice
                $customerPaket->invoices()->where('status', '!=', 'paid')->delete();
                break;
            case 'suspended':
                //return  $this->suspended_customer_paket($customerPaket);
                (new CustomerPaketAction())->update_status_customer_paket($customerPaket, 'suspended');
                break;
            case 'expired':
                //return $this->set_expired_customer_paket($customerPaket);
                (new CustomerPaketAction())->update_status_customer_paket($customerPaket, 'expired');
                break;
        }
    }

    /*
    private function cancelled_customer_paket($customerPaket)
    {
        $deleteUserMikrotik = $this->delete_user_on_mikrotik($customerPaket);
        if ($deleteUserMikrotik['success']) {
            $customerPaket->forceFill([
                'start_date' => null,
                'expired_date' => null,
                'next_billed_at' => null,
                'activation_date' => null,
                'status' => 'cancelled'
            ])->save();
            //$this->send_whatsapp_notification_to_installation_address($customerPaket, 'sendCancelledCustomerPaketNotification', null);
        }
        return $deleteUserMikrotik;
    }
        */

    /*
    private function suspended_customer_paket($customerPaket)
    {
        $disableOnMikrotik = $this->disableCustomerPaketOnMikrotik($customerPaket, 'true');
        if ($disableOnMikrotik['success']) {
            // $message = 'set to suspended';
            (new CustomerPaketAction())->update_status_customer_paket($customerPaket, 'suspended');
            //$this->send_whatsapp_notification_to_installation_address($customerPaket, 'sendSuspendedCustomerPaketNotification', null);
        }
        return $disableOnMikrotik;
    }
        */

    public function set_expired_customer_paket($customerPaket)
    {
        if ($customerPaket->isPpp()) {
            $isolirCustomerPaket = $this->isolir_secret_ppp_on_mikrotik($customerPaket);
        } else {
            $isolirCustomerPaket = $this->isolir_paket_static_on_mikrotik($customerPaket);
        }

        // $customerDetail = $customerPaket->user->user_address;
        if ($isolirCustomerPaket['success']) {
            //$message = 'set to expired';
            (new CustomerPaketAction())->update_status_customer_paket($customerPaket, 'expired');
            //$this->send_whatsapp_notification_to_installation_address($customerPaket, 'sendIsolirNotification', $this->generalLogServices::NOTIFIACTION_EXPIRED);
        }
        return $isolirCustomerPaket;
    }

    /**
     * Summary of restore_customer_paket
     * @param \App\Models\Customers\CustomerPaket $customerPaket
     * @return array{message: mixed, success: array{result: mixed, success: bool|array{success: bool}|bool}|array{message: mixed, success: bool}|array{message: string, success: bool}|array{status: array, success: bool}|array{status: mixed, success: bool}}
     */
    /*
    private function restore_customer_paket(CustomerPaket $customerPaket)
    {
        try {
            if ($customerPaket->status === 'suspended') {
                $updateMikrotikStatus = $this->disableCustomerPaketOnMikrotik($customerPaket, 'false');
                if (!$updateMikrotikStatus['success']) return $updateMikrotikStatus;
            }

            if ($customerPaket->isPpp()) {
                $updateMikrotikStatus = (new MikrotikPppService())->updateProfileSecret(
                    $customerPaket->paket->mikrotik,
                    $customerPaket->customer_ppp_paket,
                    $customerPaket->paket->paket_profile->profile_name
                );
                if (!$updateMikrotikStatus['success']) return $updateMikrotikStatus;
            } else if ($customerPaket->isIpStatic()) {
                $updateMikrotikStatus = (new MikrotikIpStaticService())->disableIpFromArp(
                    $customerPaket->paket->mikrotik,
                    $customerPaket->customer_static_paket->arp_id,
                    'false'
                );
                if (isset($updateMikrotikStatus['after']['message'])) return ['success' => false, 'message' => $updateMikrotikStatus['after']['message']];
            }

            (new CustomerPaketAction())->update_status_customer_paket($customerPaket, 'active');
            //$this->send_whatsapp_notification_to_installation_address($customerPaket, 'sendRestoreCustomerPaketNotification', null);

            return  [
                'success' => true,
                'status' => $updateMikrotikStatus
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
        */

    public function disableCustomerPaketOnMikrotik(CustomerPaket $customerPaket, $disabled)
    {
        $mikrotik = $customerPaket->paket()->withTrashed()->first()->mikrotik;

        try {
            if ($customerPaket->isPpp()) {
                $customerPppPaket = $customerPaket->customer_ppp_paket()->withTrashed()->first();
                //PPP Service
                $pppSecret = (new MikrotikPppService())->disableSecret($mikrotik, $customerPppPaket, $disabled);
                if (!$pppSecret['success']) return $pppSecret;
            } else if ($customerPaket->isIpStatic()) {
                //IP Static Service
                $customerStaticPaket = $customerPaket->customer_static_paket()->withTrashed()->first();
                $arp =  (new MikrotikIpStaticService())->disableIpFromArp($mikrotik, $customerStaticPaket->arp_id, $disabled);
                if (isset($arp['after']['message'])) return ['success' => false, 'message' => $arp['after']['message']];
            }
            return [
                'success' => true,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    public function disableWaNotificationCustomerPaketAddress(CustomerPaket $customerPaket, $input)
    {
        try {
            (new CustomerPaketAddressAction())->disableWaNotificationInstallationAddress($customerPaket,  $input);
            (new CustomerPaketAddressAction())->disableWaNotificationBillingAddress($customerPaket,  $input);
            return [
                'success' => true,
                'message' => $customerPaket->customer_installation_address->wa_notification ? trans('customer.paket.alert.wa-notification-installation-address-enable') : trans('customer.paket.alert.wa-notification-installation-address-disable')
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    public function send_whatsapp_notification_to_installation_address($customerPaket, $whatsappNotificationServiceFunction = null, $logServiceType = null)
    {
        $customerPaketInstallationAddress = $customerPaket->customer_installation_address;
        $customerHavePhoneNumber = is_null($customerPaketInstallationAddress->phone) && !$customerPaketInstallationAddress->wa_notification ? false : true;
        if ($customerHavePhoneNumber && !is_null($whatsappNotificationServiceFunction)) {
            $this->whatsappNotificationService->$whatsappNotificationServiceFunction($customerPaket);
            if (!is_null($logServiceType)) $this->generalLogServices->send_customer_notification($customerPaket, $logServiceType);
        }
        $author = Auth::user() ? Auth::user()->full_name : 'system';
        $this->whatsappNotificationService->sendAdminNotification($customerPaket->user->full_name . ' ' . $customerPaketInstallationAddress->address . ' set to ' . $customerPaket->status . ' by ' . $author);
    }

    public function commentMikrotik($expiredDate)
    {
        $commentUnpayment = $this->websystem_comment_unpayment();
        $comment = $commentUnpayment . '_' . Carbon::parse($expiredDate)->format('d_m_Y');
        return $comment;
    }


    public function isolir_secret_ppp_on_mikrotik($customerPaket)
    {
        $mikrotik = $customerPaket->paket->mikrotik;
        $autoIsolir = $mikrotik->auto_isolir;

        $profileIsolir = $autoIsolir->profile_id;
        try {
            $customerPppPaket = $customerPaket->customer_ppp_paket;
            $this->mikrotikPppService->updateProfileSecret($mikrotik, $customerPppPaket, $profileIsolir);
            return [
                'success' => true
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    public function isolir_paket_static_on_mikrotik($customerPaket)
    {
        $mikrotik = $customerPaket->paket->mikrotik;
        $autoIsolir = AutoIsolir::where('mikrotik_id', $mikrotik->id)->first();
        if (!$autoIsolir->disabled) {
            return [
                'success' => false,
                'message' => 'Isolir static paket in development'
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Isolir static paket in development'
            ];
        }
    }

    public function restore_deleted_customer_paket(CustomerPaket $customerPaket, $restoreOnMikrotik = false)
    {
        $customerPaket->restore();
        if ($customerPaket->isPpp()) {
            $customerPaket->customer_ppp_pakets()->withTrashed()->restore();
        } else if ($customerPaket->isIpStatic()) {
            $customerPaket->customer_static_pakets()->withTrashed()->restore();
        }

        $customerPaket->customer_paket_addresses()->withTrashed()->restore();
        $customerPaket->invoices()->withTrashed()->restore();

        if ($restoreOnMikrotik) {
            $this->restoreCustomerPaketOnMikrotik($customerPaket, $restoreOnMikrotik);
        } else {
            $customerPaket->forceFill([
                'status' => 'cancelled'
            ])->save();
        }

        //Enable secret on mikrotik
        $this->generalLogServices->admin_action($this->generalLogServices::RESTORE_CUSTOMER_PAKET, "Restore customer paket with invoices " . ' Server ' . $customerPaket->paket()->withTrashed()->first()->mikrotik()->withTrashed()->first()->name . ' Customer ' . $customerPaket->user()->withTrashed()->first()->full_name);
    }

    public function restoreCustomerPaketOnMikrotik($customerPaket, $restoreOnMikrotik = false)
    {
        $mikrotik = $customerPaket->mikrotik;

        if ($customerPaket->isPpp()) {
            /*PPP Paket*/
            $customerPppPaket = $customerPaket->customer_ppp_paket()->withTrashed()->first();
            if ($restoreOnMikrotik) { //Restore User Secret On Mikrotik
                $isExist = $this->mikrotikPppService->userSecretIsExist($mikrotik, $customerPppPaket->username);
                if ($isExist['success']) { //If exist, enable it
                    $customerPppPaket->forceFill([
                        'secret_id' => $isExist['user_secret']['.id']
                    ])->save();
                    return $this->disableCustomerPaketOnMikrotik($customerPaket, 'false');
                } else { //If not exist, create new user secret
                    return $this->createUserOnMikrotik($customerPaket);
                }
            }
        } else if ($customerPaket->isIpStatic()) {
            //
        }

        return ['success' => true];
    }

    public function syncronize_next_billed_at(CustomerPaket $customerPaket)
    {
        $intervalDayInvoice = $this->different_day_create_billing();
        $nextBilledAt = Carbon::parse($customerPaket->expired_date)->subDays($intervalDayInvoice);
        $customerPaket->forceFill([
            'next_billed_at' => $nextBilledAt
        ])->save();

        $this->synchronizationInvoice($customerPaket, $customerPaket->start_date, $customerPaket->expired_date, $nextBilledAt);
        /*
        $intervalDayInvoice = $this->different_day_create_billing();
        $newestInvoice = $customerPaket->invoices()->latest()->first();
        $dateNextBilledAt = Carbon::parse($customerPaket->activation_date)->subDays($intervalDayInvoice)->format('d');

        if ($newestInvoice) {
            $newestInvoicePeriod = Carbon::parse($newestInvoice->periode)->startOfMonth();
            $nowPeriod = Carbon::now()->startOfMonth();
            if ($nowPeriod->gt($newestInvoicePeriod)) {
                $nextBilledAt = Carbon::now()->setDay((int) $dateNextBilledAt);
            } else {
                $nextBilledAt = Carbon::parse($customerPaket->next_billed_at)->setDay((int) $dateNextBilledAt);
            }
        } else {
            $nextBilledAt = Carbon::now()->setDay((int) $dateNextBilledAt);
        }

        $customerPaket->forceFill([
            'next_billed_at' => $nextBilledAt
        ])->save();
        */
    }
}
