<?php

namespace App\Services;

use App\Models\User;
use App\Models\Websystem;
use App\Models\UserAddress;
use Illuminate\Support\Str;
use App\Models\Pakets\Paket;
use App\Models\Pakets\PppType;
use App\Services\PaketService;
use App\Traits\WebSystemTrait;
use Illuminate\Support\Carbon;
use App\Models\Servers\Mikrotik;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use App\Models\Pakets\PaketProfile;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;
use App\Support\CollectionPagination;
use App\Models\Customers\UserCustomer;
use App\Models\Pakets\InternetService;
use App\Models\Customers\CustomerPaket;
use App\Services\Billings\BillingService;
use App\Livewire\Actions\Users\UserAction;
use App\Models\Customers\CustomerPppPaket;
use App\Services\Billings\DeadlineService;
use App\Services\Mikrotiks\MikrotikService;
use App\Livewire\Actions\Pakets\PaketAction;
use App\Jobs\Pakets\ExportPaketToMikrotikJob;
use App\Models\Customers\CustomerPaketAddress;
use App\Jobs\Pakets\ImportPaketFromMikrotikJob;
use App\Http\Resources\Mikrotik\ProfileResource;
use App\Jobs\Customers\ExportCustomerToMikrotikJob;
use App\Jobs\Customers\ImportCustomerFromMikrotikJob;

class CustomerService
{
    use WebSystemTrait;
    private MikrotikService $mikrotikService;
    private PaketService $paketService;
    private CustomerPaketService $customerPaketService;
    private DeadlineService $deadlineService;
    protected BillingService $billingService;
    protected static ?string $password;

    public function __construct()
    {
        // Initialize
        $this->mikrotikService = new MikrotikService;
        $this->paketService = new PaketService();
        $this->customerPaketService = new CustomerPaketService;
        $this->deadlineService =  new DeadlineService;
        $this->billingService =  new BillingService;
    }


    /*
    public function neededUserSecrets(Mikrotik $mikrotik, $inCustomerManagement = false)
    {
        $mikrotikSecrets = $this->mikrotikService->getAllUserSecrets($mikrotik);
        $customerPppPakets = CustomerPppPaket::whereHas(
            'customer_paket',
            function ($builder) use ($mikrotik) {
                $builder->whereHas(
                    'paket',
                    function ($builder) use ($mikrotik) {
                        $builder->where('mikrotik_id', $mikrotik->id);
                    }
                );
            }
        )->withTrashed()->pluck('username');

        return $inCustomerManagement ? collect($mikrotikSecrets)->whereIn('name', $customerPppPakets) : collect($mikrotikSecrets)->whereNotIn('name', $customerPppPakets);
    }
*/


    public function neededExportCustomers($fromMikrotik, Mikrotik $toMikrotik, $inMikrotik = false)
    {
        $mikrotikSecrets = $this->mikrotikService->getAllUserSecrets($toMikrotik);
        $mikrotikSecrets = collect($mikrotikSecrets)->pluck('name');
        $customerPppPakets = CustomerPppPaket::whereHas(
            'customer_paket',
            function ($builder) use ($fromMikrotik) {
                $builder->whereHas(
                    'paket',
                    function ($builder) use ($fromMikrotik) {
                        $builder->where('mikrotik_id', $fromMikrotik->id);
                    }
                );
            }
        )->get();

        return $inMikrotik ? collect($customerPppPakets)->whereIn('username', $mikrotikSecrets) : collect($customerPppPakets)->whereNotIn('username', $mikrotikSecrets);
    }

    public function exportCustomersToMikrotik($fromMikrotik, Mikrotik $toMikrotik)
    {
        //export profile the required profile secret
        $neededExportProfiles = $this->paketService->neededExportProfiles($fromMikrotik, $toMikrotik);
        $neededExportCustomers = $this->neededExportCustomers($fromMikrotik, $toMikrotik);
        $pakets = Paket::whereMikrotikId($fromMikrotik->id)
            ->whereHas('customer_ppp_pakets')
            ->pluck('paket_profile_id');

        $neededExportProfiles = $neededExportProfiles->whereIn('id', $pakets);

        if (env('QUEUE_CONNECTION') == 'sync') {
            $neededExportProfiles =   (new CollectionPagination($neededExportProfiles))->collectionPaginate(Websystem::first()->max_process);
        }
        foreach ($neededExportProfiles as $neededExportProfile) {
            dispatch(new ExportPaketToMikrotikJob($toMikrotik, $neededExportProfile))->onQueue('default');
        }

        //Export Customer from Customer Management to mikrotik
        $neededExportCustomers = $this->neededExportCustomers($fromMikrotik, $toMikrotik);
        //To reduce server work processes, secret imports are limited according to settings if the queue connection is in sync mode.
        if (env('QUEUE_CONNECTION') == 'sync') {
            $neededExportCustomers =   (new CollectionPagination($neededExportCustomers))->collectionPaginate(Websystem::first()->max_process);
        }
        foreach ($neededExportCustomers as $neededExportCustomer) {
            dispatch(new ExportCustomerToMikrotikJob($toMikrotik, $neededExportCustomer))->onQueue('default');
        }
    }

    public function deleteCustomer(User $user, $deleteOnMikrotik)
    {
        $customerPakets = $user->customer_pakets()->whereNotNull('activation_date')->get();
        DB::beginTransaction();
        try {
            foreach ($customerPakets as $customerPaket) {
                if ($deleteOnMikrotik) {
                    $this->customerPaketService->delete_user_on_mikrotik($customerPaket);
                } else {
                    $this->customerPaketService->disableCustomerPaketOnMikrotik($customerPaket, 'true');
                }
            }
            (new UserAction())->delete($user);
            DB::commit();
            return ['success' => true];
        } catch (\Exception $e) {
            DB::rollBack();
            try {
                foreach ($customerPakets as $customerPaket) {
                    $this->customerPaketService->restoreCustomerPaketOnMikrotik($customerPaket, $deleteOnMikrotik);
                }
            } catch (\Exception $e) {
                return ['success' => false, 'message' => $e->getMessage()];
            }
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    public function restoreCustomer($user, $restoreOnMikrotik = false)
    {
        $user->restore();
        $customerPakets = $user->customer_pakets()->whereNotNull('activation_date')->get();
        try {
            foreach ($customerPakets as $customerPaket) {
                $this->customerPaketService->restoreCustomerPaketOnMikrotik($customerPaket, $restoreOnMikrotik);
            }
            return ['success' => true];
        } catch (\Exception $e) {
            try {
                foreach ($customerPakets as $customerPaket) {
                    if ($restoreOnMikrotik) {
                        $this->customerPaketService->delete_user_on_mikrotik($customerPaket);
                    } else {
                        $this->customerPaketService->disableCustomerPaketOnMikrotik($customerPaket, 'true');
                    }
                }
            } catch (\Exception $e) {
                return ['success' => false, 'message' => $e->getMessage()];
            }
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
}
