<?php

namespace App\Services;

use App\Models\User;
use App\Models\Websystem;
use App\Models\UserAddress;
use Illuminate\Support\Str;
use App\Models\Pakets\Paket;
use App\Models\Pakets\PppType;
use App\Services\PaketService;
use App\Traits\WebSystemTrait;
use Illuminate\Support\Carbon;
use App\Models\Servers\Mikrotik;
use App\Traits\CustomerPaketTrait;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use App\Models\Pakets\PaketProfile;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;
use App\Support\CollectionPagination;
use App\Models\Customers\UserCustomer;
use App\Models\Pakets\InternetService;
use App\Models\Customers\CustomerPaket;
use App\Services\Billings\BillingService;
use App\Livewire\Actions\Users\UserAction;
use App\Models\Customers\CustomerPppPaket;
use App\Services\Billings\DeadlineService;
use App\Services\Mikrotiks\MikrotikService;
use App\Livewire\Actions\Pakets\PaketAction;
use App\Jobs\Pakets\ExportPaketToMikrotikJob;
use App\Models\Customers\CustomerPaketAddress;
use App\Jobs\Pakets\ImportPaketFromMikrotikJob;
use App\Http\Resources\Mikrotik\ProfileResource;
use App\Jobs\Customers\ExportCustomerToMikrotikJob;
use App\Jobs\Customers\ImportCustomerFromMikrotikJob;
use App\Models\Billings\Invoice;
use App\Models\Billings\InvoiceItem;

class ImportCustomerService
{
    use WebSystemTrait, CustomerPaketTrait;
    private MikrotikService $mikrotikService;
    // private PaketService $paketService;
    // private CustomerPaketService $customerPaketService;
    private DeadlineService $deadlineService;
    protected BillingService $billingService;
    protected static ?string $password;

    public function __construct()
    {
        // Initialize
        $this->mikrotikService = new MikrotikService;
        //  $this->paketService = new PaketService();
        //  $this->customerPaketService = new CustomerPaketService;
        $this->deadlineService =  new DeadlineService;
        $this->billingService =  new BillingService;
    }

    public function importCustomersFromMikrotik(Mikrotik $mikrotik, $input)
    {
        $neededUserSecrets = $this->neededUserSecrets($mikrotik);
        $secretProfiles = collect($neededUserSecrets)->unique('profile');

        $paketProfileDontHavePaket  = PaketProfile::whereDoesntHave(
            'pakets',
            function ($pakets) use ($mikrotik) {
                $pakets->where('mikrotik_id', $mikrotik->id);
            }
        )->pluck('profile_name');
        $secretProfileNeededPakets = $secretProfiles->whereIn('profile', $paketProfileDontHavePaket)->pluck('profile');

        $mikrotikProfiles = $this->mikrotikService->getPppProfiles($mikrotik);
        $profileNeededPakets = collect($mikrotikProfiles)->whereIn('name', $secretProfileNeededPakets);

        DB::beginTransaction();
        try {
            foreach ($profileNeededPakets as $profileNeededPaket) {
                $profileNeededPaket = new ProfileResource($profileNeededPaket);
                $paketProfile = PaketProfile::whereProfileName($profileNeededPaket['name'])->first();
                $paket = (new PaketAction())->importPaket($mikrotik->id, $paketProfile, $profileNeededPaket['comment'] ?? 0);
                $paket->forceFill([
                    'mikrotik_ppp_profile_id' => $profileNeededPaket['.id']
                ])->save();
            }
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Import profile: ' . $e->getMessage());
        }

        //Import Profile needed before import customer
        $paketProfiles = PaketProfile::pluck('profile_name');
        $neededCreateProfiles = $secretProfiles->whereNotIn('profile', $paketProfiles)->pluck('profile');
        $neededCreateProfiles = collect($mikrotikProfiles)->whereIn('name', $neededCreateProfiles);

        foreach ($neededCreateProfiles as $neededCreateProfile) {
            dispatch(new ImportPaketFromMikrotikJob($mikrotik, $neededCreateProfile))->onQueue('default');
        }

        //Import secrets from mikrotik to customer management
        //To reduce server work processes, secret imports are limited according to settings if the queue connection is in sync mode.
        if (env('QUEUE_CONNECTION') == 'sync') {
            $this->importCustomerWithChunk($neededUserSecrets, $mikrotik, $input);
        } else {
            foreach ($neededUserSecrets as $neededUserSecret) {
                dispatch(new ImportCustomerFromMikrotikJob($mikrotik, $neededUserSecret, $input))->onQueue('default');
            }
        }

        return [
            'success' => true
        ];
    }

    private function importCustomerWithChunk($neededUserSecrets, $mikrotik, $input)
    {
        $renewalPeriod = $this->getRenewalPeriod($input['renewal_period']);
        $lastUser = User::orderBy('id', 'desc')->first();
        if ($lastUser) {
            $lastUserId = $lastUser->id;
        } else {
            $lastUserId = 1;
        }

        $lastCustomerPaket = CustomerPaket::orderBy('id', 'desc')->first();
        if ($lastCustomerPaket) {
            $lastCustomerPaketId = $lastCustomerPaket->id;
        } else {
            $lastCustomerPaketId = 1;
        }

        $lastInvoice = Invoice::orderBy('id', 'desc')->first();
        if ($lastInvoice) {
            $lastInvoiceId = $lastInvoice->id;
        } else {
            $lastInvoiceId = 1;
        }

        $lastUserCustomer = UserCustomer::orderBy('id', 'desc')->first();
        if ($lastUserCustomer) {
            $lastUserCustomerId = $lastUserCustomer->id;
        } else {
            $lastUserCustomerId = 1;
        }

        $insert_data_user = [];
        $insert_data_user_address = [];
        $insert_data_user_customer = [];
        $insert_data_customer_paket = [];
        $insert_data_ppp_paket = [];
        $insert_data_customer_paket_installation_address = [];
        $insert_data_customer_paket_billing_address = [];
        $insert_data_customer_role = [];
        $insert_data_invoice_customer_paket = [];
        $insert_data_item_invoice = [];

        foreach ($neededUserSecrets as $userSecret) {
            // Log::info('Last: ' . $lastUserId);
            $lastUserId = ++$lastUserId;
            // Log::info($lastUserId);
            $username = Str::slug($userSecret['name'] . Str::random(2));


            $insert_data_user[] = [
                'id' => $lastUserId,
                'username' => $username,
                'first_name' => $userSecret['name'],
                'email' => $userSecret['name'] . '_' . Str::slug(Str::random(2), '_') . '@' . $input['email_domain'],
                'password' => static::$password ??= Hash::make('password'),
                'email_verified_at' => Carbon::now()->format('Y-m-d h:i:s'),
                'disabled' => $userSecret['disabled'] === 'false' ? false : true,
            ];

            $insert_data_user_address[] = [
                'user_id' => $lastUserId,
            ];

            $lastUserCustomerId = ++$lastUserCustomerId;
            $insert_data_user_customer[] = [
                'id' => $lastUserCustomerId,
                'user_id' => $lastUserId,
            ];

            $roleId = Role::whereName('customer')->first()->id;
            $insert_data_customer_role[] = [
                'role_id' => $roleId,
                'model_type' => 'App\Models\User',
                'model_id' => $lastUserId,
            ];

            //Insert paket to customer
            $paketProfile = PaketProfile::where('profile_name', $userSecret['profile'])->first();
            $paket = $paketProfile->mikrotik_paket($mikrotik);
            $internetServiceId = InternetService::where('value', 'ppp')->first()->id;

            $startDate = $activationDate = Carbon::parse($input['activation_date']) ?? Carbon::now();
            $input['comment'] = $userSecret['comment'] ?? null;
            if ($input['use_comment_activation_date'] && !is_null($input['comment'])) {
                $commentUnpayment = Websystem::first()->comment_unpayment;
                $removed = Str::remove($commentUnpayment . '_', $input['comment']);
                $replaced = Str::replace('_', '-', $removed);
                $activationDate = Carbon::parse($replaced);

                $dayActivationDate = Carbon::parse($activationDate)->format('d');
                $startDate = Carbon::now()->setDay((int) $dayActivationDate);
                // $expiredDate = $this->deadlineService->convertSubscription($input['renewal_period'], $startDate);
            }

            $startDate = $startDate->isPast() ? $startDate : Carbon::parse($startDate)->subMonth();
            // $expiredDate = $this->deadlineService->convertSubscription($input['renewal_period'], $startDate);
            $expiredDate = Carbon::parse($startDate)->add($renewalPeriod);
            $intervalInvoiceDay = $this->different_day_create_billing();
            $nextBilled = Carbon::parse($expiredDate)->subDays($intervalInvoiceDay);

            $lastCustomerPaketId = ++$lastCustomerPaketId;
            if ($nextBilled->isPast()) {
                $lastInvoiceId = ++$lastInvoiceId;
                $mikrotikAutoIsolir = $mikrotik->auto_isolir;
                if ($mikrotikAutoIsolir->activation_date) {
                    if ($this->isPrabayar()) {
                        // dd($customerPaket->expired_date);
                        $startInvoicePeriod = Carbon::parse($expiredDate);
                        $endInvoicePeriod = Carbon::parse($startInvoicePeriod)->add($renewalPeriod);
                        $nextBilled = Carbon::parse($endInvoicePeriod)->subDays($intervalInvoiceDay);
                        $dueDate = $startInvoicePeriod;
                    } else {
                        $startInvoicePeriod = Carbon::parse($startDate);
                        $endInvoicePeriod = Carbon::parse($startInvoicePeriod)->add('1 month');
                        $nextBilled = Carbon::parse($expiredDate)->subDays($intervalInvoiceDay)->add('1 month');
                        $dueDate = $endInvoicePeriod;
                    }
                } else {
                    if ($this->isPrabayar()) {
                        $startInvoicePeriod = Carbon::parse($expiredDate);
                        $endInvoicePeriod = Carbon::parse($startInvoicePeriod)->add('1 month');
                        $nextBilled = Carbon::parse($endInvoicePeriod)->subDays($intervalInvoiceDay);
                        $dueDate = $startInvoicePeriod;
                    } else {
                        $startInvoicePeriod = Carbon::parse($startDate);
                        $endInvoicePeriod = Carbon::parse($startInvoicePeriod)->add('1 month');
                        $nextBilled = Carbon::parse($expiredDate)->subDays($intervalInvoiceDay)->add('1 month');
                        $dueDate = $endInvoicePeriod;
                    }
                }

                $invoicePeriod = Carbon::parse($startInvoicePeriod)->startOfMonth();
                $insert_data_invoice_customer_paket[] = [
                    'id' => $lastInvoiceId,
                    'user_customer_id' => $lastUserCustomerId,
                    'customer_paket_id' => $lastCustomerPaketId,
                    'periode' => $invoicePeriod->format('Y-m-d'),
                    'invoice_number' => $this->billingService->generateInvoiceNumber(),
                    'issue_date' => Carbon::now()->format('Y-m-d h:i:s'),
                    'due_date' => $dueDate->format('Y-m-d h:i:s'),
                    'start_periode' => $startInvoicePeriod->format('Y-m-d h:i:s'),
                    'end_periode' =>  $endInvoicePeriod->format('Y-m-d h:i:s'),
                    'status' => 'pending',
                ];

                $insert_data_item_invoice[] = [
                    'invoice_id' => $lastInvoiceId,
                    'item' => $paket->name . ' - ' . Carbon::parse($startInvoicePeriod)->format('d F Y') . ' - ' . Carbon::parse($endInvoicePeriod)->format('d F Y'),
                    'price' => $paket->price,
                    'discount' => 0,
                    'tax' => $this->calculateTax($paket->price)
                ];
            }

            $insert_data_customer_paket[] = [
                'id' => $lastCustomerPaketId,
                'slug' => $username,
                'user_id' => $lastUserId,
                'paket_id' => $paket->id,
                'internet_service_id' => $internetServiceId,
                'activation_date' => $input['activation_date'],
                'renewal_period' => 'monthly',
                'status' => $userSecret['disabled'] === 'false' ? 'active' : 'suspended',
                'price' => $paket->price,
                'start_date' => $startDate->format('Y-m-d h:i:s'),
                'expired_date' => $expiredDate->format('Y-m-d h:i:s'),
                'next_billed_at' => $nextBilled->format('Y-m-d h:i:s'),
            ];

            $insert_data_customer_paket_installation_address[] = [
                'customer_paket_id' => $lastCustomerPaketId,
                'address_type' => 'installation-address',
                'wa_notification' => false,
            ];

            $insert_data_customer_paket_billing_address[] = [
                'customer_paket_id' => $lastCustomerPaketId,
                'address_type' => 'billing-address',
                'wa_notification' => false,
            ];

            $insert_data_ppp_paket[] = [
                'slug' => $username,
                'customer_paket_id' => $lastCustomerPaketId,
                'username' => $userSecret['name'],
                'password_ppp' => $userSecret['password'],
                'ppp_type_id' => PppType::where('name', $userSecret['service'])->first()->id,
                'secret_id' => $userSecret['.id'],
            ];
        }

        foreach (array_chunk($insert_data_user, 500) as $key => $user) {
            User::insert($user);
        }

        foreach (array_chunk($insert_data_user_address, 500) as $key => $user_address) {
            UserAddress::insert($user_address);
        }

        foreach (array_chunk($insert_data_user_customer, 500) as $key => $user_customer) {
            UserCustomer::insert($user_customer);
        }

        foreach (array_chunk($insert_data_customer_paket, 500) as $key => $customer_paket) {
            CustomerPaket::insert($customer_paket);
        }

        foreach (array_chunk($insert_data_ppp_paket, 500) as $key => $ppp_paket) {
            CustomerPppPaket::insert($ppp_paket);
        }

        foreach (array_chunk($insert_data_customer_paket_installation_address, 500) as $key => $customer_paket_installation_address) {
            CustomerPaketAddress::insert($customer_paket_installation_address);
        }

        foreach (array_chunk($insert_data_customer_paket_billing_address, 500) as $key => $customer_paket_billing_address) {
            CustomerPaketAddress::insert($customer_paket_billing_address);
        }

        foreach (array_chunk($insert_data_customer_role, 500) as $key => $customer_role) {
            DB::table('model_has_roles')->insert($customer_role);
        }

        foreach (array_chunk($insert_data_invoice_customer_paket, 500) as $key => $invoice_customer_paket) {
            Invoice::insert($invoice_customer_paket);
        }
        foreach (array_chunk($insert_data_item_invoice, 500) as $key => $item_invoice) {
            InvoiceItem::insert($item_invoice);
        }
    }

    public function neededUserSecrets(Mikrotik $mikrotik, $inCustomerManagement = false)
    {
        $mikrotikSecrets = $this->mikrotikService->getAllUserSecrets($mikrotik);
        $customerPppPakets = CustomerPppPaket::whereHas(
            'customer_paket',
            function ($builder) use ($mikrotik) {
                $builder->whereHas(
                    'paket',
                    function ($builder) use ($mikrotik) {
                        $builder->where('mikrotik_id', $mikrotik->id);
                    }
                );
            }
        )->withTrashed()->pluck('username');

        return $inCustomerManagement ? collect($mikrotikSecrets)->whereIn('name', $customerPppPakets) : collect($mikrotikSecrets)->whereNotIn('name', $customerPppPakets);
    }
}
