<?php

namespace App\Services\Payments;

use Illuminate\Support\Str;
use App\Traits\WebSystemTrait;
use App\Models\Billings\Invoice;
use App\Models\Billings\Payment;
use App\Traits\CustomerPaketTrait;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Services\WhatsappGateway\WhatsappNotificationService;

class PartialPaymentService
{
    use WebSystemTrait, CustomerPaketTrait;
    protected $paymentService, $whatsappNotificationService;

    public function __construct(PaymentService $paymentService, WhatsappNotificationService $whatsappNotificationService)
    {
        $this->paymentService = $paymentService;
        $this->whatsappNotificationService = $whatsappNotificationService;
    }

    public function processPartialPayment(Invoice $invoice, float $amount, $paymentMethod, $bankAccount = null, $teller)
    {
        if ($amount < 0 || $amount > $invoice->remaining_amount) {
           // Log::info('Invalid partial payment amount.');
            return [
                'success' => false,
                'message' => 'Invalid partial payment amount.'
            ];
            //throw new Exception('Invalid partial payment amount.');
        }

        DB::beginTransaction();
        try {
            $customerPaket = $invoice->customer_paket;
            $user = $customerPaket->user;
            $payment = new Payment([
                'id' => Str::orderedUuid()->toString(),
                'invoice_id' => $invoice->id,
                'user_customer_id' => $user->user_customer->id,
                'customer_name' => $user->full_name,
                'customer_address' => $customerPaket->full_customer_billing_address,
                'amount' => $amount,
                'payment_date' => now(),
                'teller' => $teller,
                'payment_method' => $paymentMethod,
                'bank' => $bankAccount,
            ]);

            $paymentResult = $this->paymentService->processPayment($payment);
            if ($paymentResult['success']) {
                $payment->save();
                $this->updateInvoiceStatus($invoice);
                DB::commit();
                return ['success' => true, 'message' => 'Partial payment processed successfully.', 'payment' => $payment];
            } else {
                return [
                    'success' => false,
                    'message' => $paymentResult['message']
                ];
            }
        } catch (\Exception $e) {
            DB::rollBack();
            $invoice->forceFill(['status' => 'failed'])->save();
            return ['success' => false, 'message' => 'Partial payment failed: ' . $e->getMessage()];
        }
    }

    private function updateInvoiceStatus(Invoice $invoice)
    {
        if ($invoice->remaining_amount <= 0) {
            $invoice->status = 'paid';
        } elseif ($invoice->total_paid > 0) {
            $invoice->status = 'partially_paid';
        }
        $invoice->save();
    }
}
