<?php

namespace App\Services\Payments;

use Exception;
use Illuminate\Support\Carbon;
use App\Models\Billings\Payment;
use Illuminate\Support\Facades\Log;

/**
 * Summary of PartialPaymentService
 */
class PaymentService
{
    private $maxRetries = 3;
    private $retryDelay = 5; // seconds
    private $supportedMethods = [
        'cash',
        'paylater',
        'bank_transfer',
        'tripay',
        'midtrans'
    ];

    public function validatePaymentMethod($method)
    {
        return in_array($method, $this->supportedMethods);
    }
    public function processPayment($payment)
    {
        // $gateway = $payment->paymentGateway;
        $retries = 0;

        if (!$this->validatePaymentMethod($payment->payment_method)) {
            //throw new Exception('Unsupported payment method: ' . $payment->payment_method);
            return [
                'success' => false,
                'message' => 'Unsupported payment method: ' . $payment->payment_method
            ];
        }

        while ($retries < $this->maxRetries) {
            try {
                $result = $this->attemptPayment($payment, $payment->payment_method);

                // Trigger reconciliation after successful payment
                if ($result) {
                    app(PaymentReconciliationService::class)->reconcilePayment($payment);
                }

                Log::info('Payment processed successfully', ['payment_id' => $payment->id, 'attempt' => $retries + 1]);
                Log::info('Payment processed successfully', [
                    'payment_id' => $payment->id,
                    'attempt' => $retries + 1,
                    'method' => $payment->payment_method
                ]);

                return $result;
            } catch (Exception $e) {
                $retries++;
                Log::warning('Payment attempt failed', [
                    'payment_id' => $payment->id,
                    'attempt' => $retries,
                    'method' => $payment->payment_method,
                    'error' => $e->getMessage()
                ]);

                if ($retries >= $this->maxRetries) {
                    Log::error('Payment processing failed after max retries', [
                        'payment_id' => $payment->id,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    return [
                'success' => false,
                'message' => 'Payment processing failed after max retries. ' . $e->getMessage()
            ];
                   // throw $e;
                }

                sleep($this->retryDelay);
            }
        }
    }

    private function attemptPayment(Payment $payment, $paymentMethod)
    {
        switch ($paymentMethod) {
            case 'cash':
                return $this->processCashPayment($payment);
            case 'bank_transfer':
                return $this->processBankTransferPayment($payment);
            case 'midtrans':
                return $this->processMidtransPayment($payment);
            case 'tripay':
                return $this->processTripayPayment($payment);
            default:
                return [
                    'success' => false,
                    'message' =>  'Unsupported payment method: ' . $paymentMethod
                ];
        }
    }

    public function refundPayment(Payment $payment, float $amount)
    {
        // $gateway = $payment->paymentGateway;
        $retries = 0;

        while ($retries < $this->maxRetries) {
            try {
                $result = $this->attemptRefund($payment, $amount);
                Log::info('Refund processed successfully', [
                    'payment_id' => $payment->id,
                    'amount' => $amount,
                    'attempt' => $retries + 1
                ]);
                return $result;
            } catch (Exception $e) {
                $retries++;
                Log::warning('Refund attempt failed', [
                    'payment_id' => $payment->id,
                    'amount' => $amount,
                    'attempt' => $retries,
                    'error' => $e->getMessage()
                ]);

                if ($retries >= $this->maxRetries) {
                    Log::error('Refund processing failed after max retries', [
                        'payment_id' => $payment->id,
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    throw $e;
                }

                sleep($this->retryDelay);
            }
        }
    }

    private function attemptRefund(Payment $payment, float $amount)
    {
        switch ($payment->payment_method) {
            case 'bank_transfer':
                return $this->processGlobalRefund($payment, $amount);
            case 'cash':
                return $this->processGlobalRefund($payment, $amount);
            case 'paylater':
                return $this->processGlobalRefund($payment, $amount);
                // case 'tripay':
                //     return $this->processTripayRefund($payment, $amount);
            default:
                //throw new Exception('Unsupported payment gateway for refunds');
                Log::info('Unsupported payment gateway for refunds');
                return [
                    'success' => true,
                    'message' => 'Unsupported payment gateway for refunds'
                ];
        }
    }


    private function processCashPayment(Payment $payment)
    {
        $payment->forceFill([
            'transaction_id' => 'CASH-' . Carbon::now()->format('dmY') . strtoupper(uniqid()),
            // 'reconciliation_status' => 'completed',
        ]);
        return ['success' => true];
    }

    private function processBankTransferPayment(Payment $payment)
    {
        $payment->forceFill([
            'transaction_id' => 'TF-' . Carbon::now()->format('dmY') . strtoupper(uniqid()),
            // 'reconciliation_status' => 'completed',
        ]);
        return ['success' => true];
    }

    private function processMidtransPayment(Payment $payment)
    {
        //
    }

    private function processTripayPayment(Payment $payment)
    {
        $invoice = $payment->invoice;
        $payment->transaction_id = $invoice->order->reference;
        return ['success' => true];
    }

    private function processGlobalRefund(Payment $payment, float $amount)
    {
        // Implement PayPal refund logic here
        return [
            'success' => true,
            'message' => 'Refund processed successfully'
        ];
    }

    private function processTripayRefund(Payment $payment, float $amount)
    {
        // Implement PayPal refund logic here
        return [
            'success' => true,
            'message' => 'PayPal refund processed successfully'
        ];
    }

    /*
    private function updateInvoiceStatus(Payment $payment, $amount, $payment_method, $teller = 'tripay')
    {
       if ( Websystem::first()->isolir_driver == 'mikrotik'){
                $updateMikrotikResult = (new MikrotikPaymentService())->mikrotik_payment_comment($invoice, $comment);
                if (!$updateMikrotikResult['success']) return [
                    'success' => false,
                    'message' => throw new \Exception($updateMikrotikResult['message'])
                ];

        $payment->save();
        $totalPaid = $amount + $invoice->payments->sum('amount');
        $totalBill = $invoice->total_amount - $invoice->special_discount;
        if ($totalPaid > $totalBill) return ['success' => false, 'message' => 'Amoount greather than bill.'];
        if ($totalPaid == $totalBill) {
            $invoice->status = 'paid';
            $invoice->customer_paket->paylater_date = null;
        } elseif ($totalPaid < $totalBill) {
            if ($invoice->status != 'paylater') {
                $invoice->status = 'partially_paid';
            }
        }
        $invoice->teller_name = $teller;
        $invoice->paid_at = now();
        $invoice->save();
        $this->payment_history($invoice, $payment, 'success', $input);
        return ['success' => true];
    }

    private function updateInvoicePaylaterStatus(Invoice $invoice, $input)
    {
        //  dd($input);
        $invoice->status = 'paylater';
        $invoice->customer_paket->paylater_date = $input['paylaterDate'];
        $invoice->teller_name = Auth::user()->full_name;
        $invoice->save();
        return ['success' => true];
    }

    private function payment_history($invoice, $payment, $status, $input)
    {

        $paymentHistory = new PaymentHistory([
            'payment_id' => $invoice->id,
            'invoice_id' => $invoice->id,
            'customer_id' => $invoice->customer_paket->user->user_customer->customer_id,
            'amount' => $payment->amount ?? 0,
            'payment_method' => $input['selectedPaymentMethode'],
            'paylater_date' => $input['paylaterDate'] ?? null,
            'transaction_id' => $payment->transaction_id ?? null,
            'status' => $status,
            'notes' => $invoice->note,
        ]);
        $paymentHistory->save();
    }
        */
}
