<?php

namespace App\Services\Payments;

use Exception;
use App\Models\Billings\Invoice;
use App\Traits\WebSystemTrait;
use Illuminate\Support\Carbon;
use App\Models\Billings\Payment;
use App\Traits\CustomerPaketTrait;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\Billings\PaymentHistory;
use App\Services\CustomerPaketService;
use App\Services\WhatsappGateway\WhatsappNotificationService;

class RefundPaymentService
{
    protected $paymentService;

    public function __construct(PaymentService $paymentService)
    {
        $this->paymentService = $paymentService;
    }

    public function processRefund(Payment $payment, float $amount)
    {
        if (!$payment->isRefundable()) {
            return [
                'success' => false,
                'message' => 'This payment is not refundable.'
            ];
            // throw new Exception('This payment is not refundable.');
        }

        if ($amount > $payment->amount) {
            return [
                'success' => false,
                'message' => 'Refund amount cannot exceed the original payment amount.'
            ];
            // throw new Exception('Refund amount cannot exceed the original payment amount.');
        }
        if ($amount <= 0) {
            return [
                'success' => false,
                'message' => 'Amount field must be less than 0 .'
            ];
            // throw new Exception('Refund amount cannot exceed the original payment amount.');
        }

        DB::beginTransaction();

        try {
            $refundResult = $this->paymentService->refundPayment($payment, $amount);

            if ($refundResult['success']) {

                $payment->refunded_amount = ($payment->refunded_amount ?? 0) + $amount;
                $payment->refund_status = $payment->amount <= $payment->refunded_amount ? 'full' : 'partial';
                $payment->save();
                $paymentHistory = PaymentHistory::wherePaymentId($payment->id)->first();
                if ($paymentHistory) {
                    $paymentHistory->forceFill([
                        'amount' => $payment->amount - $payment->refunded_amount,
                        'status' => $payment->amount - $payment->refunded_amount <= 0 ? 'refunded' : 'discrepancy'
                    ])->save();
                }

                // Update invoice status if necessary
                $this->updateInvoiceStatus($payment->invoice);

                DB::commit();
                return ['success' => true, 'message' => 'Refund processed successfully.'];
            } else {
                return [
                    'success' => false,
                    'message' => $refundResult['message']
                ];
                //throw new Exception($refundResult['message']);
            }
        } catch (Exception $e) {
            DB::rollBack();
            return ['success' => false, 'message' => 'Refund failed: ' . $e->getMessage()];
        }
    }

    private function updateInvoiceStatus($invoice)
    {
        $totalPaid = $invoice->payments->sum('amount');
        $totalRefunded = $invoice->payments->sum('refunded_amount');
        $netPaid = $totalPaid - $totalRefunded;

        if ($netPaid <= 0) {
            $invoice->status = 'refunded';
        } elseif ($netPaid < $invoice->total_amount) {
            $invoice->status = 'partially_paid';
        }

        $invoice->save();
    }
}
