<?php

namespace App\Services\WhatsappGateway;

use App\Models\Bank;
use App\Models\User;
use App\Traits\Billing;
use Illuminate\Support\Str;
use App\Jobs\SendWhatsappJob;
use App\Traits\WebSystemTrait;
use Illuminate\Support\Carbon;
use App\Models\Billings\Payment;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use App\Services\WhatsappGateway\WhatsappService;
use App\Models\WhatsappGateway\WhatsappGatewayGeneral;
use App\Models\WhatsappGateway\WhatsappNotificationMessage;

class WhatsappNotificationService
{
    use Billing, WebSystemTrait;
    protected $whatsappService;
    public function __construct(
        WhatsappService $whatsappService = null
    ) {
        $this->whatsappService = $whatsappService ?? new WhatsappService();
    }

    private function delayProcess()
    {
        if (env('QUEUE_CONNECTION') === 'database') {
            return DB::table('jobs')->where('queue', 'send-whatsapp')->count() * 10;
        } else {
            return 0;
        }
    }

    private function isEnable()
    {
        return !WhatsappGatewayGeneral::first()->disabled;
    }

    private function sendWaAdmin()
    {
        return WhatsappGatewayGeneral::first()->send_wa_admin;
    }
    private function getDefaultMessage($slug)
    {
        return WhatsappNotificationMessage::whereSlug($slug)
            ->where('disabled', false)
            ->value('message');
    }

    private function getAccountBank()
    {
        $account_banks_text = '';
        $account_banks = Bank::where('disabled', false)
            ->get();

        if (count($account_banks)) {
            $i = 0;
            foreach ($account_banks as $account_bank) {
                $i++;
                $account_banks_text .= $i . '. ' . $account_bank->bank_name . ' - ' . $account_bank->account_number . ' - ' . $account_bank->account_name . '%0a';
            }
        }
        return $account_banks_text;
    }

    private function getFullBillingAddress($customerPaket)
    {
        $billingAddress = $customerPaket->customer_billing_address;
        $address = $billingAddress->address ? $billingAddress->address . ', ' : '';
        $subdistrict = $billingAddress->subDistrict ? $billingAddress->subDistrict . ', ' : '';
        $district = $billingAddress->district ? $billingAddress->district . ', ' : '';
        $city = $billingAddress->city ? $billingAddress->city . ', ' : '';
        $province = $billingAddress->province ? $billingAddress->province . ', ' : '';
        $country = $billingAddress->country ? $billingAddress->country . ', ' : '';
        return $address . $subdistrict . $district . $city . $province . $country;
    }

    private function getGender($customerPaket)
    {
        $gender = $customerPaket->user->user_customer->gender;
        return $gender ? ($gender === 'male' ? 'Bp.' : 'Ibu') : 'Bp/Ibu';
    }

    public function sendUpcomingDueReminders($customerPaket, $interval_day)
    {
        $receive = $customerPaket->customer_billing_address->phone;
        $enableNotification = $customerPaket->customer_billing_address->wa_notification;
        $message = $this->getDefaultMessage('warning_bill');

        if ($this->isEnable() && $receive && $enableNotification && $message) {
            $billing_unpayments = $customerPaket->unpayment_invoices;
            $totalBills = $this->total_bills($customerPaket);

            $pakets = $this->detail_bills($billing_unpayments, true);
            $deadline = $billing_unpayments->toQuery()->latest()->first()->due_date;
            if ($interval_day == 0) {
                $day = trans('whatsapp-gateway.wa-message.today');
            } else if ($interval_day == 1) {
                $day = trans('whatsapp-gateway.wa-message.tomorrow');
            } else {
                $day = trans('whatsapp-gateway.wa-message.days-again', ['day' => $interval_day]);
            }
            $replace = [
                '%gender%' => $this->getGender($customerPaket),
                '%name%' => $customerPaket->user->full_name,
                '%email%' => $customerPaket->user->email,
                '%customer_id%' => $customerPaket->user->user_customer->id,
                '%address%' => $customerPaket->user->user_address->address,
                '%day%' => $day,
                '%pakets%' => $pakets,
                '%paket_name%' => $billing_unpayments->first()->customer_paket->paket->name,
                '%total_bill%' => ' Rp. ' . number_format($totalBills, 2),
                '%count_bill%' => $billing_unpayments->count(),
                //'%deadline-bill%' => Carbon::parse($deadline)->format('d F Y'),
                '%deadline%' => Carbon::parse($deadline)->format('d F Y'),
                '%account_bank%' => $this->getAccountBank(),
                '%company_name%' => $this->getCompanyName(),
                '%company_address%' => $this->getCompanyAddress(),
                '%company_phone%' => $this->getCompanyPhone(),
                //'%download_invoice%' => route('download.invoice.file', $invoice->uuid),
            ];

            $message =  str_replace(array_keys($replace), $replace, $message);
            $this->send_wa($receive, $message);
        }
    }

    public function sendIsolirNotification($customerPaket)
    {
        $receive = $customerPaket->customer_billing_address->phone;
        $enableNotification = $customerPaket->customer_billing_address->wa_notification;

        if ($this->isEnable() && $receive && $enableNotification) {
            $message = $this->getDefaultMessage('isolir_paket');
            $billing_unpayments = $customerPaket->unpayment_invoices;
            if ($message && $billing_unpayments->count()) {
                $pakets = $this->detail_bills($billing_unpayments);
                $latestInvoiceUnpayment = $billing_unpayments->toQuery()->latest()->first();
                if ($latestInvoiceUnpayment) {
                    $totalBills = $this->total_bills($customerPaket);
                    $deadline = $latestInvoiceUnpayment->due_date;
                    $replace = [
                        '%gender%' => $this->getGender($customerPaket),
                        '%name%' => $customerPaket->user->full_name,
                        '%email%' => $customerPaket->user->email,
                        '%customer_id%' => $customerPaket->user->user_customer->id,
                        '%address%' => $customerPaket->user->user_address->address,
                        '%pakets%' => $pakets,
                        '%paket_name%' => $billing_unpayments->first()->customer_paket->paket->name,
                        '%total_bill%' => ' Rp. ' . number_format($totalBills, 2),
                        '%count_bill%' => $billing_unpayments->count(),
                        '%deadline%' => Carbon::parse($deadline)->format('d F Y'),
                        '%account_bank%' => $this->getAccountBank(),
                        '%company_name%' => $this->getCompanyName(),
                        '%company_address%' => $this->getCompanyAddress(),
                        '%company_phone%' => $this->getCompanyPhone()
                    ];
                    $message =  str_replace(array_keys($replace), $replace, $message);
                }
            } else {
                $message = 'Koneksi internet anda saat ini sedang bermasalah, silahkan hubungi admin kami untuk info lebih lanjut. (Kode Error: EI001)';
            }

            $this->send_wa($receive, $message);
        }
    }

    public function sendPaymentNotification($payment)
    {
        if ($this->isEnable()) {
            $customerPaket = $payment->invoice->customer_paket;
            // $receive = $customerPaket->customer_billing_address->phone;
            $billingAddress = $this->getFullBillingAddress($customerPaket);

            $replace = [
                '%gender%' => $this->getGender($customerPaket),
                '%name%' =>  $customerPaket->user->full_name,
                '%email%' => $customerPaket->user->email,
                '%address%' => $billingAddress ? $billingAddress : '-',
                '%customer_id%' => $customerPaket->user->user_customer->id,
                '%invoice_number%' =>  $payment->invoice->invoice_number,
                '%transaction_id%' =>  $payment->transaction_id,
                '%paket%' => $customerPaket->paket->name,
                '%periode%' =>  Carbon::parse($payment->invoice->start_periode)->format('d F Y - ') . Carbon::parse($payment->invoice->end_periode)->format('d F Y'),
                '%bill%' => 'Rp.' . number_format($payment->amount, 2),
                '%teller%' => $payment->teller,
                '%payment_time%' =>  Carbon::parse($payment->payment_date)->format('d F Y H:i:s'),
                '%company_name%' => $this->getCompanyName(),
                '%company_address%' => $this->getCompanyAddress(),
                '%company_phone%' => $this->getCompanyPhone()
            ];
            if ($payment['payment_method'] === 'paylater') {
                $adminMessage = $this->getDefaultMessage('notif_admin_paylater');
                $customerMessage = $this->getDefaultMessage('paylater');
                $replace['%payment_methode%'] = 'Pay Later';
                $replace['%paylater%'] = Carbon::parse($payment->paylater_date)->format('d F Y');
            } else {
                $adminMessage = $this->getDefaultMessage('notif_admin_payment');
                $customerMessage = $this->getDefaultMessage('payment');
                $paymentMethod = Str::headline($payment['payment_method']);
                $paymentMethod = $payment['payment_method'] === 'bank_transfer' ? $paymentMethod . ' ' . $payment['bank'] . '.' : ($payment['payment_method'] === 'tripay' || $payment['payment_method'] === 'midtrans' ? 'Online' : $paymentMethod . '.');
                $replace['%payment_methode%'] = $paymentMethod;
            }

            $receive = $customerPaket->customer_billing_address->phone;
            $enableNotification = $customerPaket->customer_billing_address->wa_notification;

            if ($receive && $enableNotification && $customerMessage) {
                $message =  str_replace(array_keys($replace), $replace, $customerMessage);
                $this->send_wa($receive, $message);
            }

            if ($adminMessage) {
                $adminMessage = str_replace(array_keys($replace), $replace, $adminMessage);
                $this->sendAdminNotification($adminMessage);
            }
        }
    }

    public function sendUnpaymentNotification($payment, $amount)
    {
        if ($this->isEnable()) {
            $customerPaket = $payment->invoice->customer_paket;
            $receive = $customerPaket->customer_billing_address->phone;

            $replace = [
                '%gender%' => $this->getGender($customerPaket),
                '%name%' =>  $customerPaket->user->full_name,
                '%email%' => $customerPaket->user->email,
                '%address%' => $customerPaket->user->user_address->address,
                '%customer_id%' => $customerPaket->user->user_customer->id,
                '%invoice_number%' =>  $payment->invoice->invoice_number,
                '%transaction_id%' =>  $payment->transaction_id,
                '%paket%' => $customerPaket->paket->name,
                '%periode%' =>  Carbon::parse($payment->invoice->start_periode)->format('d F Y - ') . Carbon::parse($payment->invoice->end_periode)->format('d F Y'),
                '%bill%' => 'Rp.' . number_format($amount, 2),
                '%teller%' => $payment->teller,
                '%payment_time%' =>  Carbon::parse($payment->payment_date)->format('d F Y H:i:s'),

                '%company_name%' => $this->getCompanyName(),
                '%company_address%' => $this->getCompanyAddress(),
                '%company_phone%' => $this->getCompanyPhone()
            ];

            $customerMessage = $this->getDefaultMessage('unpayment');
            $receive = $customerPaket->customer_billing_address->phone;
            $enableNotification = $customerPaket->customer_billing_address->wa_notification;

            if ($receive && $enableNotification && $customerMessage) {
                $message =  str_replace(array_keys($replace), $replace, $customerMessage);
                $this->send_wa($receive, $message);
            }

            $adminMessage = $this->getDefaultMessage('notif_admin_unpayment');
            //Log::info($adminMessage);
            if ($adminMessage) {
                $adminMessage = str_replace(array_keys($replace), $replace, $adminMessage);
                $this->sendAdminNotification($adminMessage);
            }
        }
    }

    public function sendRefundPaymentNotification($payment, $refundAmount, $custumerBankAccount = 'cash')
    {
        if ($this->isEnable()) {
            $customerPaket = $payment->invoice->customer_paket;
            $receive = $customerPaket->customer_billing_address->phone;

            $replace = [
                '%gender%' => $this->getGender($customerPaket),
                '%name%' =>  $customerPaket->user->full_name,
                '%email%' => $customerPaket->user->email,
                '%address%' => $customerPaket->user->user_address->address,
                '%customer_id%' => $customerPaket->user->user_customer->id,
                '%invoice_number%' =>  $payment->invoice->invoice_number,
                '%transaction_id%' =>  $payment->transaction_id,
                '%paket%' => $customerPaket->paket->name,
                '%periode%' =>  Carbon::parse($payment->invoice->start_periode)->format('d F Y - ') . Carbon::parse($payment->invoice->end_periode)->format('d F Y'),
                '%refund_payment%' => 'Rp.' . number_format($refundAmount, 2),
                '%teller%' => $payment->teller,
                '%customer_bank_account%' => $custumerBankAccount,
                '%payment_time%' =>  Carbon::parse($payment->payment_date)->format('d F Y H:i:s'),
                '%company_name%' => $this->getCompanyName(),
                '%company_address%' => $this->getCompanyAddress(),
                '%company_phone%' => $this->getCompanyPhone()

            ];

            $customerMessage = $this->getDefaultMessage('refund_payment');
            $receive = $customerPaket->customer_billing_address->phone;
            $enableNotification = $customerPaket->customer_billing_address->wa_notification;

            if ($receive && $enableNotification && $customerMessage) {
                $message =  str_replace(array_keys($replace), $replace, $customerMessage);
                $this->send_wa($receive, $message);
            }

            $adminMessage = $this->getDefaultMessage('notif_admin_refund_payment');
            //Log::info($adminMessage);
            if ($adminMessage) {
                $adminMessage = str_replace(array_keys($replace), $replace, $adminMessage);
                $this->sendAdminNotification($adminMessage);
            }
        }
    }

    public function sendBillsAndIsolirsToAdmin($customerBills, $customerIsolirs)
    {
        if ($this->isEnable()) {
            $adminMessage = $this->getDefaultMessage('notif_admin_bill_and_isolir');
            if ($adminMessage) {
                $replace = [
                    '%customer_bills%' =>  $customerBills,
                    '%customer_isolirs%' => $customerIsolirs,
                ];
                $adminMessage = str_replace(array_keys($replace), $replace, $adminMessage);
                $this->sendAdminNotification($adminMessage);
            }
        }
    }
    public function sendAdminNotification($message)
    {
        if ($this->sendWaAdmin()) {
            $admins = User::whereHas('user_admin')->get();
            foreach ($admins as $admin) {
                if ($admin->user_address->wa_notification && $admin->user_address->phone != null) {
                    /*dispatch(new SendWhatsappJob(
                    $admin->user_address->phone,
                    $message
                ))->onQueue('send-whatsapp')->delay($this->delayProcess());*/
                    $this->send_wa($admin->user_address->phone, $message);
                }
            }
        }
    }

    private function detail_bills($billing_unpayments, $reminderType = false)
    {
        $i = 0;
        $pakets = '';
        foreach ($billing_unpayments as $invoice) {
            //Last reminder date
            if ($reminderType) {
                $invoice->forceFill([
                    'last_reminder_date' => Carbon::now()
                ])->save();
            }

            $i++;
            $periode = Carbon::parse($invoice->start_periode)->format('d F Y - ') . Carbon::parse($invoice->end_periode)->format('d F Y');
            //$totalPayment = $billing_unpayment->payments->sum('amount');
            // $totalRefunded = $billing_unpayment->payments->sum('refunded_amount');
            // $netPaid = $totalPayment - $totalRefunded;
            // $totalBill = $billing_unpayment->amount - $netPaid;
            $totalBill = $invoice->remaining_amount;

            if ($invoice->payments()->where('payment_method', 'paylater')->count()) {
                $payment = $invoice->payments()->where('payment_method', 'paylater')->latest()->first();
                $detail_bill = trans('whatsapp-gateway.wa-message.notif-lessthan-deadline-payment-with-pay-later', [
                    'periode' => $periode,
                    'bill' => number_format($totalBill, 2),
                    'deadline' => Carbon::parse($invoice->due_date)->format('d F Y'),
                    'paylater' => Carbon::parse($payment->paylater_date)->format('d F Y'),
                    'invoice_number' => $invoice->invoice_number
                ]) . '%0a';
            } else {
                $detail_bill = trans('whatsapp-gateway.wa-message.notif-lessthan-deadline-payment', [
                    'periode' => $periode,
                    'bill' => number_format($totalBill, 2),
                    'deadline' => Carbon::parse($invoice->due_date)->format('d F Y'),
                    'invoice_number' => $invoice->invoice_number
                ]) . '%0a';
            }
            $pakets .= $i . '. ' . $detail_bill;
        }

        return $pakets;
    }
    public function total_bills($customerPaket)
    {
        /*
        $billing_unpayments = $customerPaket->unpayment_invoices;
        $totalAmountInvoices = $billing_unpayments->sum('amount');
        $totalDiscountInvoices = $billing_unpayments->sum('discount');
        $totalAmount = $totalAmountInvoices - $totalDiscountInvoices;

        $invoicePartialPaymnets = $customerPaket->partial_payments;
        $totalAmountInvoicesPartialPaymnets = $invoicePartialPaymnets->sum('amount');
        $totalPaymentRefundeds = $invoicePartialPaymnets->sum('refunded_amount');
        $totalReceivePayments = $totalAmountInvoicesPartialPaymnets - $totalPaymentRefundeds;

        return $totalAmount - $totalReceivePayments;
        */

        $invoices = $customerPaket->unpayment_invoices()->get();
        $totalAmountInvoices = $invoices->sum('amount');
        $totalDiscountInvoices = $invoices->sum('discount');
        $totalTaxInvoices = $invoices->sum('tax');

        $paymentInvoices = Payment::whereIn('invoice_id', $invoices->pluck('id'))->get();
        $totalCustomerPaketPaid = $paymentInvoices->sum('amount');
        $totalCustomerPaketRefunded = $paymentInvoices->sum('refunded_amount');
        $netCustomerPaketPaid = $totalCustomerPaketPaid - $totalCustomerPaketRefunded;


        return (($totalAmountInvoices - $totalDiscountInvoices) + $totalTaxInvoices) - $netCustomerPaketPaid;
    }

    private function send_wa($receive, $message)
    {
        Log::info('Running Job Send WA to: ' . $receive);
        dispatch(new SendWhatsappJob(
            $receive,
            $message
        ))->onQueue('send-whatsapp')->delay($this->delayProcess());
    }

    public function sendSuspendedCustomerPaketNotification($customerPaket)
    {
        if ($this->isEnable()) {
            $replace = [
                '%gender%' => $this->getGender($customerPaket),
                '%name%' => $customerPaket->user->full_name,
                '%email%' => $customerPaket->user->email,
                '%customer_id%' => $customerPaket->user->user_customer->id,
                '%address%' => $customerPaket->user->user_address->address,
                '%paket_name%' => $customerPaket->paket->name,
                '%status%' => Str::apa($customerPaket->status),
                '%company_name%' => $this->getCompanyName(),
                '%company_address%' => $this->getCompanyAddress(),
                '%company_phone%' => $this->getCompanyPhone()
            ];

            $customerMessage = $this->getDefaultMessage('disable_paket');
            $receive = $customerPaket->customer_billing_address->phone;
            $enableNotification = $customerPaket->customer_billing_address->wa_notification;
            if ($receive && $enableNotification && $customerMessage) {
                $customerMessage =  str_replace(array_keys($replace), $replace, $customerMessage);
                $this->send_wa($receive, $customerMessage);
            }
        }
    }

    public function sendRestoreCustomerPaketNotification($customerPaket)
    {
        if ($this->isEnable()) {
            $replace = [
                '%gender%' => $this->getGender($customerPaket),
                '%name%' => $customerPaket->user->full_name,
                '%email%' => $customerPaket->user->email,
                '%customer_id%' => $customerPaket->user->user_customer->id,
                '%address%' => $customerPaket->user->user_address->address,
                '%paket_name%' => $customerPaket->paket->name,
                '%bill%' => 'Rp. ' . number_format($customerPaket->price, 2),
                '%status%' => Str::apa($customerPaket->status),
                '%activation_date%' => Carbon::parse($customerPaket->activation_date)->format('d F Y'),
                '%company_name%' => $this->getCompanyName(),
                '%company_address%' => $this->getCompanyAddress(),
                '%company_phone%' => $this->getCompanyPhone()
            ];

            $customerMessage = $this->getDefaultMessage('enable_paket');
            $receive = $customerPaket->customer_billing_address->phone;
            $enableNotification = $customerPaket->customer_billing_address->wa_notification;
            if ($receive && $enableNotification && $customerMessage) {
                $customerMessage =  str_replace(array_keys($replace), $replace, $customerMessage);
                $this->send_wa($receive, $customerMessage);
            }
        }
    }

    public function sendCancelledCustomerPaketNotification($customerPaket)
    {
        if ($this->isEnable()) {
            $replace = [
                '%gender%' => $this->getGender($customerPaket),
                '%name%' => $customerPaket->user->full_name,
                '%email%' => $customerPaket->user->email,
                '%customer_id%' => $customerPaket->user->user_customer->id,
                '%address%' => $customerPaket->user->user_address->address,
                '%paket_name%' => $customerPaket->paket->name,
                '%status%' => Str::apa($customerPaket->status),
                '%company_name%' => $this->getCompanyName(),
                '%company_address%' => $this->getCompanyAddress(),
                '%company_phone%' => $this->getCompanyPhone()
            ];
            $customerMessage = $this->getDefaultMessage('cancelled_paket');
            $receive = $customerPaket->customer_billing_address->phone;
            $enableNotification = $customerPaket->customer_billing_address->wa_notification;
            if ($receive && $enableNotification && $customerMessage) {
                $customerMessage =  str_replace(array_keys($replace), $replace, $customerMessage);
                $this->send_wa($receive, $customerMessage);
            }
        }
    }

    public function sendInvoiceReadyToPaidNotification($invoice)
    {
        $customerPaket = $invoice->customer_paket;
        $receive = $customerPaket->customer_billing_address->phone;
        $enableNotification = $customerPaket->customer_billing_address->wa_notification;
        if ($this->isEnable() && $receive && $enableNotification) {
            $billing_unpayments = $customerPaket->unpayment_invoices;
            $totalBills = $this->total_bills($customerPaket);

            $pakets = $this->detail_bills($billing_unpayments, true);
            $deadline = $billing_unpayments->toQuery()->latest()->first()->due_date;
            $message = $this->getDefaultMessage('invoice_ready_to_paid');
            if ($message) {
                $filePath = $invoice->invoice_path;
                // $url = Storage::temporaryUrl($filePath, now()->addMinutes(5));


                $file_url = Storage::temporaryUrl($filePath, now()->addDay());
                $replace = [
                    '%gender%' => $this->getGender($customerPaket),
                    '%name%' => $customerPaket->user->full_name,
                    '%address%' => $customerPaket->full_customer_billing_address,
                    '%customer_id%' => $customerPaket->user->user_customer->id,

                    '%pakets%' => $pakets,
                    '%total_bill%' => ' Rp. ' . number_format($totalBills, 2),
                    //'%bill%' => 'Rp. ' . number_format($customerPaket->net_price, 2),
                    '%deadline%' => Carbon::parse($deadline)->format('d F Y'),
                    '%count_bill%' => $billing_unpayments->count(),

                    '%installation_address%' => $customerPaket->full_customer_installation_address,
                    '%status%' => Str::apa($customerPaket->status),
                    //'%download_invoice%' => route('download.invoice.file', $invoice->id),
                    '%download_invoice%' => $file_url,

                    '%company_name%' => $this->getCompanyName(),
                    '%company_address%' => $this->getCompanyAddress(),
                    '%company_phone%' => $this->getCompanyPhone()
                ];

                $message =  str_replace(array_keys($replace), $replace, $message);
                $this->send_wa($receive, $message);
            }
        }
    }

    public function sendStatusConnectionNotification($customerPaket, $lastStatus)
    {
        if ($this->isEnable()) {

            $replace = [
                '%name%' =>  $customerPaket->user->full_name,
                '%email%' => $customerPaket->user->email,
                '%address%' => $customerPaket->full_customer_installation_address,
                '%paket%' => $customerPaket->paket->name,
                '%connection_status%' => $customerPaket->online === 1 ? 'Up' : 'Down',

            ];
            $adminMessage = $this->getDefaultMessage('notif_admin_connection_status');
            if ($adminMessage && $lastStatus === $customerPaket->online) {
                $adminMessage = str_replace(array_keys($replace), $replace, $adminMessage);
                $this->sendAdminNotification($adminMessage);
            }
        }
    }
}
